/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2008 jOpenDocument, by ILM Informatique. All rights reserved.
 * 
 * The contents of this file are subject to the terms of the GNU General Public License Version 3
 * only ("GPL"). You may not use this file except in compliance with the License. You can obtain a
 * copy of the License at http://www.gnu.org/licenses/gpl-3.0.html See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each file.
 * 
 */


package org.jopendocument.model.table;

import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElements;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;
import javax.xml.bind.annotation.adapters.NormalizedStringAdapter;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import org.jopendocument.model.OpenDocument;
import org.jopendocument.model.office.OfficeForms;
import org.jopendocument.model.office.OfficeSpreadsheet;
import org.jopendocument.model.style.StyleMasterPage;
import org.jopendocument.model.style.StylePageLayout;
import org.jopendocument.model.style.StylePageLayoutProperties;
import org.jopendocument.model.style.StyleStyle;

/**
 * 
 */
@XmlAccessorType(XmlAccessType.FIELD)
@XmlType(name = "", propOrder = { "tableTableSource", "tableScenario", "officeForms", "tableShapes", "tableTableColumnsOrTableTableColumnOrTableTableColumnGroupOrTableTableHeaderColumns",
        "tableTableRowsOrTableTableRowOrTableTableRowGroupOrTableTableHeaderRows" })
@XmlRootElement(name = "table:table")
public class TableTable {

    // Une colonne ou ligne repeated est dupliquée dans la liste
    ArrayList<TableTableColumn> columns = new ArrayList<TableTableColumn>();

    ArrayList<TableTableRow> rows = new ArrayList<TableTableRow>();

    @XmlAttribute(name = "table:name")
    @XmlJavaTypeAdapter(NormalizedStringAdapter.class)
    protected String tableName;

    @XmlAttribute(name = "table:style-name")
    @XmlJavaTypeAdapter(NormalizedStringAdapter.class)
    protected String tableStyleName;

    @XmlAttribute(name = "table:protected")
    @XmlJavaTypeAdapter(NormalizedStringAdapter.class)
    protected String tableProtected;

    @XmlAttribute(name = "table:protection-key")
    @XmlJavaTypeAdapter(NormalizedStringAdapter.class)
    protected String tableProtectionKey;

    @XmlAttribute(name = "table:print-ranges")
    @XmlJavaTypeAdapter(NormalizedStringAdapter.class)
    protected String tablePrintRanges;

    @XmlAttribute(name = "table:automatic-print-range")
    @XmlJavaTypeAdapter(NormalizedStringAdapter.class)
    protected String tableAutomaticPrintRange;

    @XmlElement(name = "table:table-source")
    protected TableTableSource tableTableSource;

    @XmlElement(name = "table:scenario")
    protected TableScenario tableScenario;

    @XmlElement(name = "office:forms")
    protected OfficeForms officeForms;

    @XmlElement(name = "table:shapes")
    protected TableShapes tableShapes;

    @XmlElements( { @XmlElement(name = "table:table-columns", required = true, type = TableTableColumns.class),
            @XmlElement(name = "table:table-column", required = true, type = TableTableColumn.class),
            @XmlElement(name = "table:table-column-group", required = true, type = TableTableColumnGroup.class),
            @XmlElement(name = "table:table-header-columns", required = true, type = TableTableHeaderColumns.class) })
    protected List<Object> tableTableColumnsOrTableTableColumnOrTableTableColumnGroupOrTableTableHeaderColumns;

    @XmlElements( { @XmlElement(name = "table:table-rows", required = true, type = TableTableRows.class), @XmlElement(name = "table:table-row", required = true, type = TableTableRow.class),
            @XmlElement(name = "table:table-row-group", required = true, type = TableTableRowGroup.class),
            @XmlElement(name = "table:table-header-rows", required = true, type = TableTableHeaderRows.class) })
    protected List<Object> tableTableRowsOrTableTableRowOrTableTableRowGroupOrTableTableHeaderRows;

    private int printStartRow = 0;

    private int printStartCol = 0;

    private int printStopRow = 0;

    private int printStopCol = 0;

    private OfficeSpreadsheet spreadsheet;

    /**
     * Gets the value of the tableName property.
     * 
     * @return possible object is {@link String }
     * 
     */
    public String getTableName() {
        return tableName;
    }

    /**
     * Sets the value of the tableName property.
     * 
     * @param value allowed object is {@link String }
     * 
     */
    public void setTableName(String value) {
        this.tableName = value;
    }

    /**
     * Gets the value of the tableStyleName property.
     * 
     * @return possible object is {@link String }
     * 
     */
    public String getTableStyleName() {
        return tableStyleName;
    }

    /**
     * Sets the value of the tableStyleName property.
     * 
     * @param value allowed object is {@link String }
     * 
     */
    public void setTableStyleName(String value) {
        this.tableStyleName = value;
    }

    /**
     * Gets the value of the tableProtected property.
     * 
     * @return possible object is {@link String }
     * 
     */
    public String getTableProtected() {
        if (tableProtected == null) {
            return "false";
        } else {
            return tableProtected;
        }
    }

    /**
     * Sets the value of the tableProtected property.
     * 
     * @param value allowed object is {@link String }
     * 
     */
    public void setTableProtected(String value) {
        this.tableProtected = value;
    }

    /**
     * Gets the value of the tableProtectionKey property.
     * 
     * @return possible object is {@link String }
     * 
     */
    public String getTableProtectionKey() {
        return tableProtectionKey;
    }

    /**
     * Sets the value of the tableProtectionKey property.
     * 
     * @param value allowed object is {@link String }
     * 
     */
    public void setTableProtectionKey(String value) {
        this.tableProtectionKey = value;
    }

    /**
     * Sets the value of the tablePrintRanges property.
     * 
     * @param value allowed object is {@link String }
     * 
     */
    public void setTablePrintRanges(String value) {
        if (value == null) {
            throw new IllegalArgumentException("ranges is null");
        }
        this.tablePrintRanges = value;
        int s = value.indexOf(':');
        String l = value.substring(0, s);
        String r = value.substring(s + 1);

        String vl = l.substring(l.indexOf('.') + 1);
        String vr = r.substring(r.indexOf('.') + 1);

        {

            int j = 0;
            for (int i = vl.length() - 1; i >= 0; i--) {
                int c = vl.charAt(i);
                if (Character.isLetter(c)) {

                    int val = c - 'A';
                    printStartCol += (val * Math.pow(26, j));
                    j++;
                } else {
                    printStartRow = i;

                }
            }
            String substring = vl.substring(vl.length() - printStartRow);
            printStartRow = Integer.valueOf(substring) - 1;
        }

        {

            int j = 0;
            for (int i = vr.length() - 1; i >= 0; i--) {
                int c = vr.charAt(i);
                if (Character.isLetter(c)) {

                    int val = c - 'A';
                    printStopCol += (val * Math.pow(26, j));
                    j++;
                } else {
                    printStopRow = i;

                }
            }
            String substring = vr.substring(vl.length() - printStopRow);
            printStopRow = Integer.valueOf(substring) - 1;
        }

    }

    public int getPrintStartCol() {
        return printStartCol;
    }

    public int getPrintStartRow() {
        return printStartRow;
    }

    public int getPrintStopCol() {
        return printStopCol;
    }

    public int getPrintStopRow() {
        return printStopRow;
    }

    /**
     * Gets the value of the tableAutomaticPrintRange property.
     * 
     * @return possible object is {@link String }
     * 
     */
    public String getTableAutomaticPrintRange() {
        return tableAutomaticPrintRange;
    }

    /**
     * Sets the value of the tableAutomaticPrintRange property.
     * 
     * @param value allowed object is {@link String }
     * 
     */
    public void setTableAutomaticPrintRange(String value) {
        this.tableAutomaticPrintRange = value;
    }

    /**
     * Gets the value of the tableTableSource property.
     * 
     * @return possible object is {@link TableTableSource }
     * 
     */
    public TableTableSource getTableTableSource() {
        return tableTableSource;
    }

    /**
     * Sets the value of the tableTableSource property.
     * 
     * @param value allowed object is {@link TableTableSource }
     * 
     */
    public void setTableTableSource(TableTableSource value) {
        this.tableTableSource = value;
    }

    /**
     * Gets the value of the tableScenario property.
     * 
     * @return possible object is {@link TableScenario }
     * 
     */
    public TableScenario getTableScenario() {
        return tableScenario;
    }

    /**
     * Sets the value of the tableScenario property.
     * 
     * @param value allowed object is {@link TableScenario }
     * 
     */
    public void setTableScenario(TableScenario value) {
        this.tableScenario = value;
    }

    /**
     * Gets the value of the officeForms property.
     * 
     * @return possible object is {@link OfficeForms }
     * 
     */
    public OfficeForms getOfficeForms() {
        return officeForms;
    }

    /**
     * Sets the value of the officeForms property.
     * 
     * @param value allowed object is {@link OfficeForms }
     * 
     */
    public void setOfficeForms(OfficeForms value) {
        this.officeForms = value;
    }

    /**
     * Gets the value of the tableShapes property.
     * 
     * @return possible object is {@link TableShapes }
     * 
     */
    public TableShapes getTableShapes() {
        return tableShapes;
    }

    /**
     * Sets the value of the tableShapes property.
     * 
     * @param value allowed object is {@link TableShapes }
     * 
     */
    public void setTableShapes(TableShapes value) {
        this.tableShapes = value;
    }

    public void addColumn(TableTableColumn col) {
        for (int i = 0; i < col.getTableNumberColumnsRepeated(); i++) {
            this.columns.add(col);
        }

        col.setTable(this);

    }

    public void addRow(TableTableRow r) {
        for (int i = 0; i < r.getTableNumberRowsRepeated(); i++) {
            this.rows.add(r);
        }
        r.setTable(this);
    }

    public List<TableTableColumn> getColumns() {
        return columns;
    }

    /**
     * Return all the rows (duplicated if repeated)
     */
    public List<TableTableRow> getRows() {
        return rows;
    }

    @Override
    public String toString() {

        return "TableTable: print:" + this.getPrintStartCol() + "," + this.getPrintStartRow() + " : " + this.getPrintStopCol() + "," + this.getPrintStopRow();
    }

    public StyleStyle getColumnStyle(String s) {

        return spreadsheet.getBody().getDocument().getAutomaticStyles().getColumnStyle(s);
    }

    public StyleStyle getCellStyle(String s) {

        return spreadsheet.getBody().getDocument().getAutomaticStyles().getCellStyle(s);
    }

    public StyleStyle getRowStyle(String s) {
        return spreadsheet.getBody().getDocument().getAutomaticStyles().getRowStyle(s);
    }

    public void setSpreadsheet(OfficeSpreadsheet s) {
        spreadsheet = s;

    }

    public OfficeSpreadsheet getSpreadsheet() {
        return spreadsheet;
    }

    public List<TableTableColumn> getColumnsInRange(int startCol, int stopCol) {
        List<TableTableColumn> colsInRange = new Vector<TableTableColumn>(stopCol - startCol + 1);
        for (int i = startCol; i <= stopCol; i++) {
            colsInRange.add(columns.get(i));
        }
        return colsInRange;
    }

    public List<TableTableRow> getRowsInRange(int startRow, int stopRow) {
       List<TableTableRow> rowsInRange = new ArrayList<TableTableRow>(stopRow - startRow + 1);
        for (int i = startRow; i <= stopRow; i++) {
            if (i < rows.size())
                rowsInRange.add(rows.get(i));
        }
        return rowsInRange;
    }

    public int getPrintWidth() {
        int w = 0;
        List<TableTableColumn> cols = getColumnsInRange(getPrintStartCol(), getPrintStopCol());
        for (TableTableColumn col : cols) {
            w += col.getWidth();
        }
        return w;
    }

    public int getPrintWidth(double resizeFactor) {
        int w = 0;
        List<TableTableColumn> cols = getColumnsInRange(getPrintStartCol(), getPrintStopCol());
        for (TableTableColumn col : cols) {
            w += col.getWidth() / resizeFactor;
        }
        return w;
    }

    public TableTableColumn getColumn(int i) {
        return this.columns.get(i);
    }

    public int getPrintHeight() {
        int h = 0;

        List<TableTableRow> rowsInRange = getRowsInRange(getPrintStartRow(), getPrintStopRow());
        for (TableTableRow row : rowsInRange) {
            h += row.getHeight();
        }
        return h;
    }

    public int getPrintHeight(double resizeFactor) {
        int h = 0;

        List<TableTableRow> rowsInRange = getRowsInRange(getPrintStartRow(), getPrintStopRow());
        for (TableTableRow row : rowsInRange) {
            h += row.getHeight() / resizeFactor;

        }

        return h;
    }

    public int getHeight(int startRow, int stopRow) {
        int h = 0;
        for (int i = startRow; i <= stopRow; i++) {
            final TableTableRow tableTableRow = rows.get(i);
            h += tableTableRow.getHeight();
        }
        return h;
    }

    Vector<TableTableColumn> allColumns;

    public TableTableColumn getColumnAtPosition(int colPosition) {
        // FIXME: faire idem rows
        if (allColumns == null) {
            allColumns = new Vector<TableTableColumn>();
            for (int i = 0; i < this.columns.size(); i++) {
                TableTableColumn col = this.getColumn(i);
                for (int n = 0; n < col.getTableNumberColumnsRepeated(); n++) {
                    allColumns.add(col);
                }

            }

        }
        return allColumns.get(colPosition);

    }

    /**
     * Returns the page layout properties
     * 
     * @return the properties
     */
    public StylePageLayoutProperties getPageLayoutProperties() {
        if (tableStyleName == null) {
            return null;
        }
        // Getting page layout... a vibrant story
        OpenDocument doc = this.getSpreadsheet().getBody().getDocument();

        StyleStyle tableStyle = doc.getAutomaticStyles().getTableStyle(tableStyleName);

        String styleMasterPageName = tableStyle.getMasterPageName();
        StyleMasterPage styleMasterPage = doc.getMasterStyles().getMasterPageFromStyleName(styleMasterPageName);
        String pageLayoutName = styleMasterPage.getStylePageLayoutName();
        StylePageLayout pageLayout = doc.getAutomaticStyles().getStylePageLayoutFromStyleName(pageLayoutName);
        return pageLayout.getPageLayoutProperties();
    }

}
