/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2008 jOpenDocument, by ILM Informatique. All rights reserved.
 * 
 * The contents of this file are subject to the terms of the GNU
 * General Public License Version 3 only ("GPL").  
 * You may not use this file except in compliance with the License. 
 * You can obtain a copy of the License at http://www.gnu.org/licenses/gpl-3.0.html
 * See the License for the specific language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each file.
 * 
 */

package org.jopendocument.dom;

import org.jopendocument.dom.OOSingleXMLDocument;
import org.jopendocument.dom.OOXMLDocument;
import org.jopendocument.util.CopyUtils;
import org.jopendocument.util.ExceptionUtils;
import org.jopendocument.util.StreamUtils;
import org.jopendocument.util.StringInputStream;
import org.jopendocument.util.Zip;
import org.jopendocument.util.ZippedFilesProcessor;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

/**
 * An OpenDocument package, ie a zip containing XML documents and their associated files.
 * 
 * @author ILM Informatique 2 août 2004
 */
public class ODPackage {

    // use raw format, otherwise spaces are added to every spreadsheet cell
    private static final XMLOutputter OUTPUTTER = new XMLOutputter(Format.getRawFormat());

    private static final Set<String> subdocNames;
    static {
        subdocNames = new HashSet<String>();
        // section 2.1 of OpenDocument-v1.1-os.odt
        subdocNames.add("content.xml");
        subdocNames.add("styles.xml");
        subdocNames.add("meta.xml");
        subdocNames.add("settings.xml");
    }

    // values are either byte[] or OOXMLDocument
    private final Map<String, Object> files;
    private ContentTypeVersioned type;

    private ODPackage() {
        this.files = new HashMap<String, Object>();
        this.type = null;
    }

    public ODPackage(Map<String, Object> files) {
        this();

        for (final Map.Entry<String, Object> e : files.entrySet()) {
            this.putFile(e.getKey(), e.getValue());
        }
    }

    public ODPackage(InputStream ins) throws IOException {
        this();

        final ByteArrayOutputStream out = new ByteArrayOutputStream(4096);
        new ZippedFilesProcessor() {
            @Override
            protected void processEntry(String name, InputStream in) throws IOException {
                final Object res;
                if (subdocNames.contains(name)) {
                    try {
                        res = new OOXMLDocument(OOUtils.getBuilder().build(in));
                    } catch (JDOMException e) {
                        // always correct
                        throw ExceptionUtils.createExn(IllegalStateException.class, "parse error", e);
                    }
                } else {
                    out.reset();
                    StreamUtils.copy(in, out);
                    res = out.toByteArray();
                }
                putFile(name, res);
            }
        }.process(ins);
    }

    public ODPackage(ODPackage o) {
        this();
        // ATTN this works because, all files are read upfront
        for (final Map.Entry<String, Object> f : o.getFiles().entrySet()) {
            final String name = f.getKey();
            final Object data = f.getValue();
            // assume byte[] are immutable
            if (data instanceof byte[])
                this.putFile(name, data);
            else
                this.putFile(name, CopyUtils.copy(data));
        }
    }

    /**
     * The version of this package, <code>null</code> if it cannot be found (eg this package is
     * empty, or contains no xml).
     * 
     * @return the version of this package, can be <code>null</code>.
     */
    public final String getVersion() {
        if (this.getContent() == null)
            return null;
        else
            return this.getContent().getVersion();
    }

    /**
     * The type of this package, <code>null</code> if it cannot be found (eg this package is
     * empty).
     * 
     * @return the type of this package, can be <code>null</code>.
     */
    public final ContentTypeVersioned getContentType() {
        if (this.type == null) {
            if (this.files.containsKey("mimetype"))
                this.type = ContentTypeVersioned.fromMime(new String(this.getBinaryFile("mimetype")));
            else if (this.getVersion().equals(OOUtils.VERSION_1)) {
                final Element contentRoot = this.getContent().getDocument().getRootElement();
                final String docClass = contentRoot.getAttributeValue("class", contentRoot.getNamespace("office"));
                this.type = ContentTypeVersioned.fromClass(docClass);
            } else if (this.getVersion().equals(OOUtils.VERSION_2)) {
                final Element bodyChild = (Element) this.getContent().getChild("body").getChildren().get(0);
                this.type = ContentTypeVersioned.fromBody(bodyChild.getName());
            }
        }
        return this.type;
    }

    public final String getMimeType() {
        return this.getContentType().getMimeType();
    }

    // *** getter on files

    protected final Map<String, Object> getFiles() {
        return Collections.unmodifiableMap(this.files);
    }

    public byte[] getBinaryFile(String entry) {
        return (byte[]) this.files.get(entry);
    }

    public OOXMLDocument getXMLFile(String xmlEntry) {
        return (OOXMLDocument) this.files.get(xmlEntry);
    }

    public final OOXMLDocument getContent() {
        return this.getXMLFile("content.xml");
    }

    /**
     * Return an XML document.
     * 
     * @param xmlEntry the filename, eg "styles.xml".
     * @return the matching document, or <code>null</code> if there's none.
     * @throws JDOMException if error about the XML.
     * @throws IOException if an error occurs while reading the file.
     */
    public Document getDocument(String xmlEntry) {
        return this.getXMLFile(xmlEntry).getDocument();
    }

    // *** setter

    public void putFile(String entry, Object data) {
        if (data == null) {
            this.files.remove(entry);
        } else {
            if (subdocNames.contains(entry)) {
                final OOXMLDocument oodoc;
                if (data instanceof Document)
                    oodoc = new OOXMLDocument((Document) data);
                else
                    oodoc = (OOXMLDocument) data;
                // si le package est vide n'importe quelle version convient
                if (this.getVersion() != null && !oodoc.getVersion().equals(this.getVersion()))
                    throw new IllegalArgumentException("version mismatch " + this.getVersion() + " != " + oodoc);
                data = oodoc;
            } else if (!(data instanceof byte[]))
                throw new IllegalArgumentException("should be byte[] " + data);
            this.files.put(entry, data);
        }
    }

    /**
     * Transform this to use a {@link OOSingleXMLDocument}. Ie after this method, only
     * "content.xml" remains and it's an instance of OOSingleXMLDocument.
     * 
     * @return the created OOSingleXMLDocument.
     */
    public OOSingleXMLDocument toSingle() {
        // this removes xml files used by OOSingleXMLDocument
        final Document content = removeAndGetDoc("content.xml");
        final Document styles = removeAndGetDoc("styles.xml");
        final Document settings = removeAndGetDoc("settings.xml");

        final OOSingleXMLDocument single = OOSingleXMLDocument.createFromDocument(content, styles, settings, this.files);
        this.putFile("content.xml", single);
        return single;
    }

    private Document removeAndGetDoc(String name) {
        final OOXMLDocument xmlDoc = (OOXMLDocument) this.files.remove(name);
        return xmlDoc == null ? null : xmlDoc.getDocument();
    }

    // *** save

    public final void save(OutputStream out) throws IOException {
        final Zip z = new Zip(out);

        final Manifest manifest = new Manifest(this.getVersion(), this.getMimeType());
        for (final String name : this.files.keySet()) {
            // added at the end
            if (name.equals("mimetype") || name.equals(Manifest.ENTRY_NAME))
                continue;

            final Object val = this.files.get(name);
            final OutputStream o = z.createEntry(name);
            if (val instanceof OOXMLDocument) {
                OUTPUTTER.output(((OOXMLDocument) val).getDocument(), o);
                manifest.addEntry(name, "text/xml");
            } else {
                StreamUtils.copy(new ByteArrayInputStream((byte[]) val), o);
            }
            o.close();
        }

        z.zip("mimetype", new StringInputStream(this.getMimeType()));
        z.zip(Manifest.ENTRY_NAME, new StringInputStream(manifest.asString()));
        z.close();
    }

}