/*
 * libgsmat: An implementation of GSM 07.07
 *
 * Written by Klaus-Peter Junghanns <support@junghanns.net>
 *
 * Copyright (C) 2005, Junghanns.NET GmbH
 * All Rights Reserved.
 *
 * Parts taken from libpri-1.0.9
 * Written by Mark Spencer <markster@linux-support.net>
 *
 * Copyright (C) 2001, Linux Support Services, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */
 
#ifndef _LIBGSM_H
#define _LIBGSM_H

/* Modem types */
#define GSM_MODEM_MOTOROLA_G20 		0

#define GSM_DEBUG_NONE			0
#define GSM_DEBUG_AT			(1 << 0)

/* GSM D-Channel Events */
#define GSM_EVENT_DCHAN_UP	 	1	/* MUX is up */
#define GSM_EVENT_DCHAN_DOWN 		2	/* MUX is down */
#define GSM_EVENT_RING		 	3	/* Incoming call */
#define GSM_EVENT_HANGUP	 	4	/* Call got hung up */
#define GSM_EVENT_ALERTING	 	5	/* Call is ringing (alerting) */
#define GSM_EVENT_ANSWER	 	6	/* Call has been answered */
#define GSM_EVENT_PIN_REQUIRED	 	7
#define GSM_EVENT_SM_RECEIVED	 	8	/* A short message has been received */
#define GSM_EVENT_ERROR		 	9
#define GSM_EVENT_PUK_REQUIRED	 	10

//typedef struct gsm_call gsm_call;

typedef struct gsm_call gsm_call;

typedef struct gsm_event_generic {
	/* Events with no additional information fall in this category */
	int e;
} gsm_event_generic;

typedef struct gsm_event_error {
	int e;
	char err[256];
	int cause;
	int hard;
} gsm_event_error;

typedef struct gsm_event_ringing {
	int e;
	int channel;
	gsm_call *call;
} gsm_event_ringing;

typedef struct gsm_event_answer {
	int e;
	int channel;
	gsm_call *call;
} gsm_event_answer;

typedef struct gsm_event_ring {
	int e;
	int channel;				/* Channel requested */
	int callingpres;			/* Presentation of Calling CallerID */
	char callingnum[256];		/* Calling number, network provided */
	char callingname[256];		/* Calling name (if provided) */
	char callednum[256];		/* Called number */
	gsm_call *call;			/* Opaque call pointer */
} gsm_event_ring;

typedef struct gsm_event_hangup {
	int e;
	int channel;				/* Channel requested */
	int cause;
	gsm_call *call;			/* Opaque call pointer */
} gsm_event_hangup;	

typedef struct gsm_event_sm_received {
	int e;
	char pdu[255];			/* short message in PDU format */
	char sender[255];
	char smsc[255];
	int len;
	char text[255];
} gsm_event_sm_received;	

typedef union {
	int e;
	gsm_event_generic gen;			/* Generic view */
	gsm_event_error	  error;			/* Error view */
	gsm_event_ring	  ring;			/* Ring */
	gsm_event_hangup  hangup;		/* Hang up */
	gsm_event_ringing ringing;		/* Ringing */
	gsm_event_answer  answer;		/* Answer */
	gsm_event_sm_received sm_received; 	/* SM RX */
} gsm_event;

struct gsm_modul;


/* Create a D-channel/MUX on a given file descriptor.  The file descriptor must be a
   channel operating in HDLC mode with FCS computed by the fd's driver.  Also it
   must be NON-BLOCKING! Frames received on the fd should include FCS.  
   modemtype must be one of the GSM_MODEM_* types.
   */
extern struct gsm_modul *gsm_new(int fd, int modemtype, char *pin, int span, int channel);

/* Set debug parameters on a gsm modul  */
extern void gsm_set_debug(struct gsm_modul *gsm, int debug);

/* Set file descriptor for debugging to a file */
extern void gsm_set_debug_fd(struct gsm_modul *gsm, int fd);

/* Run GSM on the given D-channel/MUX, taking care of any events that
   need to be handled.  If block is set, it will block until an event
   occurs which needs to be handled */
extern gsm_event *gsm_dchannel_run(struct gsm_modul *gsm, int block);

/* Check for an outstanding event on the GSM */
gsm_event *gsm_check_event(struct gsm_modul *gsm, int doread);

/* Give a name to a given event ID */
extern char *gsm_event2str(int id);

/* Give a name to a node type */
extern char *gsm_modem2str(int id);

/* Print an event */
extern void gsm_dump_event(struct gsm_modul *gsm, gsm_event *e);

/* Answer the call on the given channel (ignored if you called acknowledge already). */

extern int gsm_answer(struct gsm_modul *gsm);

/* Hangup a call */
extern int gsm_hangup(struct gsm_modul *gsm);

extern void gsm_destroycall(struct gsm_modul *gsm, gsm_call *call);

/* Create a new call */
extern gsm_call *gsm_new_call(struct gsm_modul *gsm);

extern void gsm_send_pin(struct gsm_modul *gsm, char *pin);

/* How long until you need to poll for a new event */
extern struct timeval *gsm_schedule_next(struct gsm_modul *gsm);

/* Run any pending schedule events */
extern gsm_event *gsm_schedule_run(struct gsm_modul *gsm);

extern int gsm_dial(struct gsm_modul *gsm, int clir, char *called);

extern int gsm_sms_send_text(struct gsm_modul *gsm, char *destination, char *msg);
extern int gsm_sms_send_pdu(struct gsm_modul *gsm, char *pdu);

extern int gsm_request_status(struct gsm_modul *gsm);

extern int gsm_restart(struct gsm_modul *gsm, int when);

extern int gsm_wait(struct gsm_modul *gsm);

extern int gsm_available(struct gsm_modul *gsm);

extern int gsm_poweroff(struct gsm_modul *gsm);


/* Override message and error stuff */
extern void gsm_set_message(void (*__gsm_error)(char *, int));
extern void gsm_set_error(void (*__gsm_error)(char *, int));

/* Get file descriptor */
extern int gsm_fd(struct gsm_modul *gsm);

#endif
