/*
 * libgsmat: An implementation of GSM 07.07
 *
 * Written by Klaus-Peter Junghanns <support@junghanns.net>
 *
 * Copyright (C) 2005, Junghanns.NET GmbH
 * All Rights Reserved.
 *
 * Parts taken from libpri-1.0.9
 * Written by Mark Spencer <markster@linux-support.net>
 *
 * Copyright (C) 2001, Linux Support Services, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */
 
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/select.h>
#include <stdarg.h>
#include <math.h>
#include "libgsmat.h"
#include "gsm_internal.h"
#include "gsm_modem.h"

char *gsm_modem2str(int node)
{
	switch(node) {
	case GSM_MODEM_MOTOROLA_G20:
		return "Motorola g20";
	default:
		return "Invalid value";
	}
}

void gsm_send_pin(struct gsm_modul *gsm, char *pin) {
    gsm_modem_send_pin(gsm, pin);
}

struct gsm_modul *gsm_new(int fd, int modemtype, char *pin, int span, int channel)
{
	struct gsm_modul *p;
	p = malloc(sizeof(struct gsm_modul));
	if (p) {
		memset(p, 0, sizeof(struct gsm_modul));
		p->fd = fd;
		p->span = span;
		p->channel = channel;
		p->modemtype = modemtype;
		p->echo = 1;
		p->sanidx = 0;
		p->lastres = 0;
		p->dchanup = 0;
		p->restart_timer = 0;
		p->state = GSM_STATE_POWER_ON;
		p->schedev = 0;
		p->restart_timer = 0;
		/* Start MUX layer */
		gsm_modem_start(p);
	}
	return p;
}

int gsm_restart(struct gsm_modul *p, int when)
{
    if (p) {
	if (p->debug)
	    gsm_message(p, "scheduling restart\n");
	if (!p->restart_timer)
	    p->restart_timer = gsm_schedule_event(p, when, gsm_modem_restart, p);
        return 0;
    }
    return -1;
}

int gsm_wait(struct gsm_modul *p)
{
    if (p) {
	while ((p->state == GSM_STATE_SM_SEND_REQ) || (p->state == GSM_STATE_SM_SENDING)) {
	    usleep(300);
	}
        return 0;
    }
    return -1;
}

int gsm_poweroff(struct gsm_modul *p)
{
    return gsm_modem_poweroff(p);
}

int gsm_available(struct gsm_modul *p) {
    if (p->debug)
	gsm_message(p, "state = %d\n", p->state);
    if (p->state == GSM_STATE_READY) {
	return 1;
    } else {
	return 0;
    }
}

char *gsm_event2str(int id)
{
	switch(id) {
	case GSM_EVENT_DCHAN_UP:
		return "D-Channel Up";
	case GSM_EVENT_DCHAN_DOWN:
		return "D-channel Down";
	case GSM_EVENT_RING:
		return "Ring";
	default:
		return "Unknown Event";
	}
}

gsm_event *gsm_check_event(struct gsm_modul *gsm, int doread)
{
	char buf[1024];
	int res = 0;
	gsm_event *e;
	
	if (doread) {
//	    gsm_error(gsm, "check event\n");
	    res = read(gsm->fd, buf, sizeof(buf));
	    if (res < 0) {
		if (errno != EAGAIN)
			gsm_error(gsm, "Read on %d failed: %s\n", gsm->fd, strerror(errno));
		return NULL;
	    }
	}
	e = gsm_modem_receive(gsm, buf, res);
	return e;
}

static int wait_gsm(struct gsm_modul *gsm)
{	
	struct timeval *tv, real;
	fd_set fds;
	int res;
	FD_ZERO(&fds);
	FD_SET(gsm->fd, &fds);
	tv = gsm_schedule_next(gsm);
	if (tv) {
		gettimeofday(&real, NULL);
		real.tv_sec = tv->tv_sec - real.tv_sec;
		real.tv_usec = tv->tv_usec - real.tv_usec;
		if (real.tv_usec < 0) {
			real.tv_usec += 1000000;
			real.tv_sec -= 1;
		}
		if (real.tv_sec < 0) {
			real.tv_sec = 0;
			real.tv_usec = 0;
		}
	}
	res = select(gsm->fd + 1, &fds, NULL, NULL, tv ? &real : tv);
	if (res < 0) 
		return -1;
	return res;
}

gsm_event *gsm_mkerror(struct gsm_modul *gsm, char *errstr)
{
	/* Return a configuration error */
//	gsm->ev.err.e = GSM_EVENT_CONFIG_ERR;
	strncpy(gsm->ev.error.err, errstr, sizeof(gsm->ev.error.err) - 1);
	return &gsm->ev;
}


gsm_event *gsm_dchannel_run(struct gsm_modul *gsm, int block)
{
	gsm_event *e;
	int res;
	if (!gsm)
		return NULL;
	if (block) {
		do {
			e =  NULL;
			res = wait_gsm(gsm);
			/* Check for error / interruption */
			if (res < 0) 
				return NULL;
			if (!res) {
			    e = gsm_schedule_run(gsm);
			} else {
				e = gsm_check_event(gsm,1);
			}
		} while(!e);
	} else {
		e = gsm_check_event(gsm, 1);
		return e;
	}
	return e;
}

void gsm_set_debug(struct gsm_modul *gsm, int debug)
{
	if (!gsm)
		return;
	gsm->debug = debug;
}

void gsm_set_debug_fd(struct gsm_modul *gsm, int fd)
{
	if (!gsm)
		return;
	gsm->debugfd = fd;
}

void gsm_destroycall(struct gsm_modul *gsm, gsm_call *call)
{
//	if (pri && call)
//		__q931_destroycall(pri, call);
	return;
}

int gsm_answer(struct gsm_modul *gsm)
{
	if (!gsm)
		return -1;
	return gsm_modem_answer(gsm);
}

int gsm_hangup(struct gsm_modul *gsm)
{
	if (!gsm)
		return -1;
	return gsm_modem_hangup(gsm);
}

gsm_call *gsm_new_call(struct gsm_modul *gsm)
{
	if (!gsm)
		return NULL;
//	return q931_new_call(pri);
	return NULL;
}

void gsm_dump_event(struct gsm_modul *gsm, gsm_event *e)
{
	if (!gsm || !e)
		return;
	gsm_message(gsm, "Event type: %s (%d)\n", gsm_event2str(e->gen.e), e->gen.e);
}


int gsm_dial(struct gsm_modul *gsm, int clir, char *called)
{
	if (!gsm)
		return -1;
	return gsm_modem_dial(gsm, clir, called);
}	

int gsm_sms_send_text(struct gsm_modul *gsm, char *destination, char *msg) {
	if (!gsm)
		return -1;
	return gsm_modem_sms_send_text(gsm, destination, msg);
}

int gsm_sms_send_pdu(struct gsm_modul *gsm, char *pdu) {
	if (!gsm)
		return -1;
	return gsm_modem_sms_send_pdu(gsm, pdu);
}

int gsm_request_status(struct gsm_modul *p) {
	if (!p)
		return -1;
	if (!p->restart_timer)
	    p->restart_timer = gsm_schedule_event(p, 1000, gsm_modem_restart, p);
	return gsm_modem_request_status(p);
}

static void (*__gsm_error)(char *stuff, int span);
static void (*__gsm_message)(char *stuff, int span);

void gsm_set_message(void (*func)(char *stuff, int channel))
{
	__gsm_message = func;
}

void gsm_set_error(void (*func)(char *stuff, int channel))
{
	__gsm_error = func;
}

void gsm_message(struct gsm_modul *gsm,char *fmt, ...)
{
	char tmp[1024];
	va_list ap;
	va_start(ap, fmt);
	vsnprintf(tmp, sizeof(tmp), fmt, ap);
	va_end(ap);
	if (__gsm_message)
		__gsm_message(tmp, gsm->channel);
	else
		fputs(tmp, stdout);
}

void gsm_error(struct gsm_modul *gsm,char *fmt, ...)
{
	char tmp[1024];
	va_list ap;
	va_start(ap, fmt);
	vsnprintf(tmp, sizeof(tmp), fmt, ap);
	va_end(ap);
	if (__gsm_error)
		__gsm_error(tmp, gsm->channel);
	else
		fputs(tmp, stderr);
}

int gsm_fd(struct gsm_modul *gsm)
{
	return gsm->fd;
}


