/*
   Fritz!Box Remote CAPI implementation for Linux (Fritz!Box Fon Remote CAPI).
   Copyright (C) 2009 Marco Zissen <maz@v3v.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation;

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.
   IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) AND AUTHOR(S) BE LIABLE FOR ANY
   CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES
   WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
   ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
   OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

   ALL LIABILITY, INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PATENTS,
   COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS, RELATING TO USE OF THIS
   SOFTWARE IS DISCLAIMED.
*/
#include <linux/kernel.h>
#include <linux/module.h>

#include "fbrcapi.h"

#define CAPI_INTEROPERABILITY		0x20

#define CAPI_FUNCTION_REGISTER		0
#define CAPI_FUNCTION_RELEASE		1
#define CAPI_FUNCTION_GET_PROFILE	2

static unsigned char send_buf[8192];

#ifdef HAVE_KZALLOC
#if (HAVE_KZALLOC = 0)
static inline void *kzalloc(size_t size, int flags)
{
	void *ptr;
	ptr = kmalloc(size, flags);
	if(!ptr)
	{
		return ptr;
	}
	memset(ptr, 0, size);
	return ptr;
}
#endif
#endif

static struct fbrc_application *fbrc_application_add(struct fbrc_session *session, __u16 appl, __u16 kapplid)
{
	struct fbrc_application *app = kzalloc(sizeof(*app), GFP_KERNEL);

	fbrc_debug(3, "fbrc_application_add(session %p app %p applid %d kapplid %d)", session, app, appl, kapplid);

	if (!app)
		return NULL;

	app->applid = appl;
	app->kapplid = kapplid;
	app->registered = 0;
	
	list_add_tail(&app->list, &session->applications);

	return app;
}

static void fbrc_application_del(struct fbrc_session *session, struct fbrc_application *app)
{	
	fbrc_debug(3, "fbrc_application_del(session %p app %p)", session, app);
	
	if (app) {
		list_del(&app->list);
		kfree(app);
	}
}

static struct fbrc_application *fbrc_application_get(struct fbrc_session *session, int pattern, __u16 value)
{
	struct fbrc_application *app;
	struct list_head *p, *n;

	list_for_each_safe(p, n, &session->applications)
	{
		app = list_entry(p, struct fbrc_application, list);
		
		switch (pattern)
		{				
			// check for applid 
			case 0:
				if (app->applid == value)
					return app;
				break;
			
			// check for kernel applid 
			case 1:
				if (app->kapplid == value)
					return app;
				break;
		}
	}

	return NULL;
}

static int fbrc_controller_get(struct fbrc_session *session, int pattern, __u16 value)
{
	int i;
	
	switch(pattern)
	{
		// check local controller, get fritz!box controller
		case 0:			
			for(i=0; i<MAX_CONTROLLERS; i++)
			{
				if(session->ctrl[i].cnr == value)
					return i+1;
			}			
			break;
		
		// check fritz!box controller, get local controller
		case 1:	
			if((value-1) >= 0 && (value-1) < MAX_CONTROLLERS)
				return session->ctrl[value-1].cnr;
	}
	
	return 0;
}

static unsigned fbrc_capi_handle_interop_msg(struct fbrc_session *session, struct sk_buff *skb)
{
	struct capi_profile cprofile;
	struct fbrc_application *app;         	
	int Ctrl = 0;
	__u16 RApplID = 0xffff;
	__u16 func = CAPIMSG_U16(skb->data, 13);
	
	switch(func)
	{
		case CAPI_FUNCTION_REGISTER:		
			app = fbrc_application_get(session, 0, RApplID);
			if(app) 
			{
				app->applid = ((skb->data[2] | (skb->data[3]  << 8)));
				app->registered = 1;
				fbrc_debug(2, "REGISTER : session %p applid %d kapplid %d", session, app->applid, app->kapplid);
			} 
			else 
			{
				fbrc_debug(1, "REGISTER : Failed! Application not found!");
			}
			
			fbrc_debug(3, "REGISTER : %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d", skb->data[0], skb->data[1], skb->data[2], skb->data[3], skb->data[4], skb->data[5], skb->data[6], skb->data[7], skb->data[8], skb->data[9], skb->data[10], skb->data[11], skb->data[12], skb->data[13], skb->data[14], skb->data[15], skb->data[16], skb->data[17], skb->data[18], skb->data[19], skb->data[20], skb->data[21], skb->data[22], skb->data[23], skb->data[24], skb->data[25]);
			break;
			
		case CAPI_FUNCTION_RELEASE:
			fbrc_debug(3, "RELEASE  : %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d", skb->data[0], skb->data[1], skb->data[2], skb->data[3], skb->data[4], skb->data[5], skb->data[6], skb->data[7], skb->data[8], skb->data[9], skb->data[10], skb->data[11], skb->data[12], skb->data[13], skb->data[14], skb->data[15], skb->data[16], skb->data[17], skb->data[18], skb->data[19], skb->data[20], skb->data[21], skb->data[22], skb->data[23], skb->data[24], skb->data[25]);			
			break;
		
		case CAPI_FUNCTION_GET_PROFILE:			
			memcpy(&cprofile, skb->data+18, sizeof(struct capi_profile));
			Ctrl = session->controllers;
			
			fbrc_debug(3, "PROFILE  : %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d", skb->data[0], skb->data[1], skb->data[2], skb->data[3], skb->data[4], skb->data[5], skb->data[6], skb->data[7], skb->data[8], skb->data[9], skb->data[10], skb->data[11], skb->data[12], skb->data[13], skb->data[14], skb->data[15], skb->data[16], skb->data[17], skb->data[18], skb->data[19], skb->data[20], skb->data[21], skb->data[22], skb->data[23], skb->data[24], skb->data[25]);
			
			if(!Ctrl) {
				memcpy(&session->profile, &cprofile, sizeof(struct capi_profile));
				session->controllers = cprofile.ncontroller;
			}
			else if (Ctrl > 0)
			{				
				capi_version fbrc_version = { FB_CAPI_MAJOR, FB_CAPI_MINOR, FB_MANU_MAJOR, FB_MANU_MINOR };
				Ctrl = fbrc_controller_get(session, 0, Ctrl);
				memcpy(&session->ctrl[Ctrl-1].profile, &cprofile, sizeof(session->ctrl[Ctrl-1].profile));
				memcpy(&session->ctrl[Ctrl-1].manu, FB_CAPI_MANUF, sizeof(session->ctrl[Ctrl-1].manu));
				memcpy(session->ctrl[Ctrl-1].serial, FB_CAPI_SERIAL, sizeof(session->ctrl[Ctrl-1].serial));
				memcpy(&session->ctrl[Ctrl-1].version, &fbrc_version, sizeof(capi_version));
				
				if(session->controllers != session->profile.ncontroller)
					session->controllers++;					
			}	
			break;
			
		default:
			fbrc_debug(1, "Unable to handle interop request %d. Function not supported!", func);
			fbrc_debug(3, "UNKNOWN  : %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d", skb->data[0], skb->data[1], skb->data[2], skb->data[3], skb->data[4], skb->data[5], skb->data[6], skb->data[7], skb->data[8], skb->data[9], skb->data[10], skb->data[11], skb->data[12], skb->data[13], skb->data[14], skb->data[15], skb->data[16], skb->data[17], skb->data[18], skb->data[19], skb->data[20], skb->data[21], skb->data[22], skb->data[23], skb->data[24], skb->data[25]);						
			break;
	}

	wake_up_interruptible(&session->wq);	
	return 0;
}

static unsigned fbrc_capi_build_interop_msg(int msg_type, unsigned applid, unsigned ctrl, unsigned b3con, unsigned b3blks, unsigned b3size, char *msg)
{	
	unsigned char param[16];
	unsigned msg_len = 0;	
	unsigned param1_len = 0;
	unsigned param2_len = 0;	
	unsigned mapplid = applid;

	memset(param, 0, sizeof(param));

	switch(msg_type)
	{
		case CAPI_FUNCTION_REGISTER:
			msg_len=0x14;			
			param1_len=0x09; param2_len=0x06;
			mapplid=0;			
			capimsg_setu16(param, 0, b3con);
			capimsg_setu16(param, 2, b3blks);
			capimsg_setu16(param, 4, b3size);
			break;
		case CAPI_FUNCTION_RELEASE:
			msg_len=0x0e;			
			param1_len=0x03; param2_len=0x00;
			capimsg_setu16(param, 0, 0x00);
			break;
		case CAPI_FUNCTION_GET_PROFILE:
			msg_len=0x12;			
			param1_len=0x05; param2_len=0x02;
			mapplid=0;	
			capimsg_setu16(param, 0, ctrl);
			capimsg_setu16(param, 2, 0x00);		
			break;
		default:
			fbrc_debug(1, "Error: unsupported interop message %d (ignored)", msg_type);
			return 0;
	}

	// CAPI-Over-TCP Header
	msg[0] = 0x80;				        // 0x80 = This is a message
	capimsg_setu16(msg, 1, msg_len);		// Message-Length
	
	// CAPI-Over-TCP Message
	capimsg_setu16(msg, 3, msg_len);		// Message-Length
	capimsg_setu16(msg, 5, mapplid);		// Application-ID

	msg[7] = CAPI_INTEROPERABILITY;        		// Command (interoperability!)
	msg[8] = CAPI_REQ;        			// Subcommand

	capimsg_setu16(msg, 9, 0x01);			// Msg-Nr
	capimsg_setu16(msg, 11, 0x02);			// Default for interop: 0x02 0x00
	msg[13] = param1_len;				// Length (Param1)
	capimsg_setu16(msg, 14, msg_type);		// Interop type
	msg[16] = param2_len;       			// Length (Param2)
	
	memcpy(msg+17, param, param1_len);		// Parameters

	return msg_len+3;
}

unsigned fbrc_capi_register(struct fbrc_session *session, __u16 maxb3con, __u16 maxb3blks, __u16 maxb3size, __u16 applid)
{
	unsigned char msg[128];
	struct fbrc_application *app = NULL;	
	__u16 RApplID = 0xffff;
	int msg_len = 0;
	
	if(!session->sock) {
		if(fbrc_init_socket(session) < 0)
			return -EIO;
	}

	app = fbrc_application_get(session, 1, applid);
	if(app) {
		fbrc_debug(4, "Already registered (ignoring applid %d).", applid);	
		return 0;
	}

	msg_len = fbrc_capi_build_interop_msg(CAPI_FUNCTION_REGISTER, 0, 0, maxb3con, maxb3blks, maxb3size, msg);

	RApplID = 0xffff;
	app = fbrc_application_add(session, RApplID, applid);
	if(fbrc_send_socket(session, msg, msg_len) > 0)	
		fbrc_schedule(session);
	
	wait_event_interruptible_timeout(session->wq, app->registered, INTERRUPTIBLE_TIMEOUT);
	if(!app->registered) {
		fbrc_application_del(session, app);
		return -ETIMEDOUT;    	
	}

	return 0;
}

unsigned fbrc_capi_release(struct fbrc_session *session, __u16 applid)
{
	unsigned char msg[128];
	struct fbrc_application *app;
	int msg_len = 0;

	app = fbrc_application_get(session, 1, applid);
	if(app == NULL)
		return -EFAULT;

	msg_len = fbrc_capi_build_interop_msg(CAPI_FUNCTION_RELEASE, app->applid, 0, 0, 0, 0, msg);
	if(msg_len) {
		if(fbrc_send_socket(session, msg, msg_len) > 0)
			fbrc_schedule(session);		
			
		fbrc_debug(2, "RELEASE  : session %p applid %d kapplid %d", session, app->applid, app->kapplid);
	}

	fbrc_application_del(session, app);
	return 0;

}

unsigned fbrc_capi_get_profile(struct fbrc_session *session, unsigned ctrl)
{
	unsigned char msg[128];	
	int msg_len = 0;

	msg_len = fbrc_capi_build_interop_msg(CAPI_FUNCTION_GET_PROFILE, 0, ctrl, 0, 0, 0, msg);
	if(msg_len) {
		if(fbrc_send_socket(session, msg, msg_len) > 0)
			fbrc_schedule(session);
		
		return 0;
	}
	
	return -EIO;
}

unsigned fbrc_capi_send_message(struct fbrc_session *session, struct sk_buff *skb)
{
	struct fbrc_application *app;
	int cmd, subcmd, ctrl, len, ncci;
	int datalen = 0;
	__u16 applid;

	if(skb->len <= 0)
		return -EIO;
		
	applid = CAPIMSG_APPID(skb->data);
	app = fbrc_application_get(session, 1, applid);
	if(!app)
		return -EFAULT;
	
	// remap ApplID and controller
	cmd = CAPIMSG_COMMAND(skb->data);
	subcmd = CAPIMSG_SUBCOMMAND(skb->data);		
	ctrl = CAPIMSG_CONTROLLER(skb->data);
	ncci = CAPIMSG_NCCI(skb->data);
	len = (skb->data[0] | (skb->data[1] << 8));

	CAPIMSG_SETAPPID(skb->data, app->applid);		
	skb->data[8] = fbrc_controller_get(session, 0, ctrl);
				
	// Copy the complete message after the third byte
	memcpy(send_buf + 3, skb->data, sizeof(send_buf) - 3);

	// capi over tcp header (0x80, 2 bytes length)
	send_buf[0] = 0x80;
	send_buf[1] = send_buf[3];
	send_buf[2] = send_buf[4];

	// DATA_B3_REQ specific:
	// we have to copy  the buffer and patch the buffer address!
	if(cmd == CAPI_DATA_B3 && subcmd == CAPI_REQ)
	{
		void *dataptr;
		u_int32_t data;

		// Data length
		datalen = (send_buf[19] | (send_buf[20] << 8));

		// get the address of the buffer
		if(sizeof(void *) != 4)
		{
			// 64 bit
			if (len >= 30)
			{
				u_int64_t data64;
				
				memcpy(&data64, send_buf + 25, sizeof(u_int64_t));
				if (data64 != 0) dataptr = (void *)(unsigned long)data64;
				else dataptr = send_buf + len + 3;
			}
			else {
				dataptr = send_buf + len + 3;
			}
		}
		else
		{
			// 32 bit
			memcpy(&data, send_buf+15, 4);
			if (data != 0) dataptr = (void *)((u_int32_t)data);
			else dataptr = send_buf + len + 3;
		}

		// copy buffer to this address
		memcpy(send_buf+(len+3), (unsigned char*)dataptr, datalen);
		len += datalen;

		// !!! set buffer address to zero !!!
		send_buf[15] = send_buf[16] =  send_buf[17] = send_buf[18] = 0x00;

		// set (whole !) message length (with payload)
		send_buf[1] = len & 0xff;
		send_buf[2] = (len >> 8) & 0xff;
	}

	if(fbrc_send_socket(session, send_buf, len + 3) > 0)
	{		
		fbrc_schedule(session);
		fbrc_debug(4, "fbrc_capi_send_message(applid %d ctrl %d cmd %d subcmd %d)", applid, ctrl, cmd, subcmd);
		return 0;
	}
	
	return -EIO;
}

unsigned fbrc_capi_recv_message(struct fbrc_session *session, struct sk_buff *skb)
{
	struct fbrc_application *app;
	unsigned applid = 0;
	unsigned ctrl = 0;
	unsigned new_ctrl = 0;
	__u16 cmd, subcmd, ncci;
	
	// No data? return!
	if(skb->len <= 0)
		return -EIO;

	cmd = CAPIMSG_COMMAND(skb->data);
	subcmd = CAPIMSG_SUBCOMMAND(skb->data);
		
	// Handle interop message
	if(cmd == CAPI_INTEROPERABILITY && subcmd == CAPI_CONF) {
		fbrc_capi_handle_interop_msg(session, skb);
		return 0;
	}

	// Remap applid & ctrl and handle capi message
	applid = (skb->data[2] | (skb->data[3] << 8));
	ncci = CAPIMSG_NCCI(skb->data);	
	ctrl = CAPIMSG_CONTROLLER(skb->data);
	new_ctrl = fbrc_controller_get(session, 1, ctrl);
			
	app = fbrc_application_get(session, 0, applid);
	if(app && new_ctrl > 0 && new_ctrl <= session->controllers)	
	{
		// set new applid & ctrl
		CAPIMSG_SETAPPID(skb->data, app->kapplid);		
		skb->data[8] = new_ctrl;
				
		capi_ctr_handle_message(&session->ctrl[ctrl - 1], app->kapplid, skb);
		
		fbrc_debug(4, "fbrc_capi_recv_message(applid %d ctrl %d cmd %d subcmd %d)", applid, new_ctrl, cmd, subcmd);
		return 0;
	}
	
	return -EFAULT;
}
