/*
   Fritz!Box Remote CAPI implementation for Linux (Fritz!Box Fon Remote CAPI).
   Copyright (C) 2009 Marco Zissen <maz@v3v.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation;

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.
   IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) AND AUTHOR(S) BE LIABLE FOR ANY
   CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES
   WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
   ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
   OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

   ALL LIABILITY, INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PATENTS,
   COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS, RELATING TO USE OF THIS
   SOFTWARE IS DISCLAIMED.
*/
#include <linux/kernel.h>
#include <linux/module.h>

#include "fbrcapi.h"

static struct fbrc_session fbrc_s;

static unsigned char addr[64];						// Module-Param addr=X (FB Host/IP)
static unsigned int port = FB_DEFAULT_PORT;			// Module-Param port=X (FB-Remote CAPI Port (default 5031))
static unsigned int debug = 1;						// Module-Param debug=X (Debug-Level 0-3)
static unsigned int boost = 1;						// Module-Param boost=[1]0] (renice kernel thread on/off)
static unsigned int controllers[MAX_CONTROLLERS];	// Module-Param controllers=1,2,3,4,...
static unsigned int controllers_argc = 0;			// Module-Param controllers (argument count)

static DECLARE_COMPLETION(on_exit);

static void fbrc_register_appl(struct capi_ctr *ctrl, __u16 appl, capi_register_params *rp)
{
	fbrc_debug(4, "fbrc_register_appl(ctrl %p appl %d rp %p)", ctrl, appl, rp);	
	fbrc_capi_register((struct fbrc_session *)ctrl->driverdata, rp->level3cnt, rp->datablkcnt, rp->datablklen, appl);
}

static void fbrc_release_appl(struct capi_ctr *ctrl, __u16 appl)
{
	fbrc_debug(4, "fbrc_release_appl(ctrl %p appl %d)", ctrl, appl);	
	fbrc_capi_release((struct fbrc_session *)ctrl->driverdata, appl);
}

static int fbrc_load_firmware(struct capi_ctr *ctrl, capiloaddata *data)
{
	fbrc_debug(4, "fbrc_load_firmware(ctrl %p data %p)", ctrl, data);
	return 0;
}

static void fbrc_reset_ctr(struct capi_ctr *ctrl)
{		
	struct fbrc_session *session = ctrl->driverdata;

	fbrc_debug(4, "fbrc_reset_ctr(ctrl %p)", ctrl);
		
	capi_ctr_reseted(ctrl);

	atomic_inc(&session->terminate);
	fbrc_schedule(session);
}

static char *fbrc_procinfo(struct capi_ctr *ctrl)
{
	fbrc_debug(4, "fbrc_procinfo(%d)", ctrl);
	return MNAME_LONG;
}

static u16 fbrc_send_message(struct capi_ctr *ctrl, struct sk_buff *skb)
{	
	u16 ret = fbrc_capi_send_message((struct fbrc_session *)ctrl->driverdata, skb);
	if(skb) 
		kfree_skb(skb);
	return ret;
}

static int fbrc_recv_message(void *data)
{
	wait_queue_t wait;
	struct fbrc_session *session = data;
	struct sock *sk = session->sock->sk;
	struct sk_buff *skb;
		
	fbrc_debug(3, "fbrc_recv_message() thread started ...");
	daemonize(MNAME);	
	if(boost) set_user_nice(current, -15);
	
	init_waitqueue_entry(&wait, current);
	add_wait_queue(sk->sk_sleep, &wait);
			
	while (!atomic_read(&session->terminate))
	{	
		set_current_state(TASK_INTERRUPTIBLE);	
		
		while ((skb = skb_dequeue(&sk->sk_receive_queue)))
		{
			__u8 hdr, hdrlen;
			__u16 len;

			skb_orphan(skb);

			if (sk->sk_state != TCP_ESTABLISHED)
				break;

			while (skb->len > 0)
			{
				hdr = skb->data[0];

				// Message type
				switch (hdr & 0xc0)
				{
					case 0x40:
						hdrlen = 2;
						len = skb->data[1];
						break;
					case 0x80:
						hdrlen = 3;
						len = skb->data[1] | (skb->data[2] << 8);
						break;
					default:
						hdrlen = 1;
						len = 0;
						break;
				}

				fbrc_debug(4, "hdr 0x%02x hdrlen %d len %d skblen %d skbdatalen %d", hdr, hdrlen, len, skb->len, skb->data_len);

				if (hdrlen + len > skb->len) {
					fbrc_debug(1, "Wrong size or header information in frame!");
					break;
				}

				if (len == 0) {
					fbrc_debug(1, "Incompatible header received!");
					skb_pull(skb, hdrlen);			
					continue;
				}

				// Check for non-linear skb
				if(skb_is_nonlinear(skb)) 
				{
				    #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,18)
					if (skb_linearize(skb, GFP_ATOMIC) != 0) 
				    #else
					if (skb_linearize(skb) != 0)
				    #endif					
					{
					
						fbrc_debug(1, "Error: No memory to linearize!");
						break;			                
					}				
				}
				
				// Send skb to capi app
				skb_pull(skb, hdrlen);								
				fbrc_capi_recv_message(session, skb_clone(skb, GFP_ATOMIC));				
				skb_pull(skb, len);	
			}

			kfree_skb(skb);
		}
		
		schedule();
	}

	set_current_state(TASK_RUNNING);	
	remove_wait_queue(sk->sk_sleep, &wait);	
	if(boost) set_user_nice(current, +15);
	
	fbrc_debug(3, "fbrc_recv_message() thread stopped!");
	
	complete_and_exit(&on_exit, 0);
}



static int fbrc_ctr_read_proc(char *page, char **start, off_t off, int count, int *eof, struct capi_ctr *ctrl)
{	
	struct fbrc_session *session = ctrl->driverdata;
	struct fbrc_application *app;
	struct list_head *p, *n;
	int len = 0;
		
	len += sprintf(page + len, "%s\n\n", fbrc_procinfo(ctrl));
	len += sprintf(page + len, "addr %s\n", ctrl->name);
	len += sprintf(page + len, "ctrl %d\n", session->controllers);

	list_for_each_safe(p, n, &session->applications) {
		app = list_entry(p, struct fbrc_application, list);
		len += sprintf(page + len, "appl %d -> %d\n", app->applid, app->kapplid);
	}

	if (off + count >= len)
		*eof = 1;

	if (len < off)
		return 0;

	*start = page + off;

	return ((count < len - off) ? count : len - off);
}

static int fbrc_load_profile(struct fbrc_session *session)
{
	int ret = 0;
	int i, j;
		
	// load profile
	session->controllers = 0;
		
	// fetch controller count
	ret = fbrc_capi_get_profile(session, 0);
	if(ret) {
		fbrc_debug(1, "Error: Unable to fetch CAPI profile!");
		return ret;
	}

	ret = wait_event_interruptible_timeout(session->wq, session->controllers, INTERRUPTIBLE_TIMEOUT);
	if(ret)
		fbrc_debug(1, "%d CAPI controllers found!", session->controllers);	
	else
		fbrc_debug(1, "Unable to fetch CAPI controller count (%d)!", session->controllers);	
		
	if (!ret) return -ETIMEDOUT;
	session->controllers = 1;
		
	// fetch profile
	for(i=0; i<session->profile.ncontroller; i++)
	{		
		int use_controller = 1;	
		char cname[64];
				
		if(i >= MAX_CONTROLLERS)
			break;
		
		// use controller? Module-Param: controllers=1,2,3,4
		if(controllers_argc > 0)
		{				
			use_controller = 0;
			
			for(j = 0; j < controllers_argc; j++)
				if(controllers[j] == (i+1))
					use_controller = 1;			
		}
		
		session->ctrl[i].owner      = THIS_MODULE;
		session->ctrl[i].driverdata = session;
		snprintf(cname, sizeof(cname), "%s_ctrl_%d", MNAME, i+1);
		strcpy(session->ctrl[i].name, cname);

		session->ctrl[i].driver_name   = MNAME;
		session->ctrl[i].load_firmware = fbrc_load_firmware;
		session->ctrl[i].reset_ctr     = fbrc_reset_ctr;
		session->ctrl[i].register_appl = fbrc_register_appl;
		session->ctrl[i].release_appl  = fbrc_release_appl;
		session->ctrl[i].send_message  = fbrc_send_message;
		session->ctrl[i].procinfo      = fbrc_procinfo;
		session->ctrl[i].ctr_read_proc = fbrc_ctr_read_proc;
		session->ctrl[i].cnr = 0;
		
		if(use_controller)
		{
			ret = attach_capi_ctr(&session->ctrl[i]);
			if (ret < 0) {
				fbrc_debug(1, "Unable to attach controller \"%s\" (error %d)!", cname, ret);
				return -EBUSY;
			}
			
			fbrc_debug(2, "Attaching local controller #%d --> remote controller #%d", session->ctrl[i].cnr, i+1);
			
			ret = fbrc_capi_get_profile(session, i+1);
			ret = wait_event_interruptible_timeout(session->wq, session->controllers, INTERRUPTIBLE_TIMEOUT);
			if(!ret)
				return -ETIMEDOUT;
			
			capi_ctr_ready(&session->ctrl[i]);
		}
	}	
			
	return 0;
}

static int fbrc_attach_device(void)
{	
	int ret = 0;

	// init socket	
	ret = fbrc_init_socket(&fbrc_s);
	if(ret < 0) {		
		return ret;
	}
	
	// load profile
	ret = fbrc_load_profile(&fbrc_s);
	if(ret) {
		fbrc_debug(1, "Unable to load profile (error %d)!", ret);
		return ret;
	}
	 	
	return ret;
}

int fbrc_stop_kthread(void)
{
	// release kthread
	fbrc_debug(3, "Stopping kthread() ...");	
	if(fbrc_s.thread_id) {		
		atomic_inc(&fbrc_s.terminate);		
		fbrc_schedule(&fbrc_s);
		wait_for_completion(&on_exit);
	}
	
	fbrc_s.thread_id = 0;
	return 0;
}

int fbrc_start_kthread(void)
{
	// start kthread for receiving
	fbrc_debug(3, "Starting kthread() ...");	
	atomic_set(&fbrc_s.terminate, 0);	
	fbrc_s.thread_id = kernel_thread(fbrc_recv_message, &fbrc_s, CLONE_KERNEL);
	if(fbrc_s.thread_id == 0) {		
		return -EIO;
	}
	
	return 0;
}

static int __init fbrc_init(void)
{			
	fbrc_debug(0, "%s %s %s %s", MNAME_LONG, MVERSION, MCOPYRIGHT, MAUTHOR);
		
	// Initialisation		
	memset(&fbrc_s.profile, 0, sizeof(fbrc_s.profile));
	atomic_set(&fbrc_s.terminate, 0);
	fbrc_s.thread_id = 0;	
	fbrc_s.sock = NULL;
		
	INIT_LIST_HEAD(&fbrc_s.applications);
	init_waitqueue_head(&fbrc_s.wq);
			
	// module parameters
	if(!strlen(addr)) strncpy(addr, FB_DEFAULT_HOST, sizeof(addr));
	memcpy(fbrc_s.addr, addr, sizeof(fbrc_s.addr));
	fbrc_s.port = port;
	fbrc_s.debug = debug;	
	
	// is addr an IP?
	{
		int i1, i2, i3, i4;
		if(sscanf(addr, "%d.%d.%d.%d", &i1, &i2, &i3, &i4) != 4)
		{
			fbrc_debug(1, "%s is not an IP address!", addr);
			return -EIO;
		}
	}		
		
	return fbrc_attach_device();		
}

static void __exit fbrc_exit(void)
{	
	int i;
	
	fbrc_debug(1, "Removing Fritz!Box remote CAPI controllers ...");		
	
	// release kthread
	fbrc_stop_kthread();

	// detach controllers
	for(i=0; i<MAX_CONTROLLERS; i++)		
		if(fbrc_s.ctrl[i].cnr != 0)
			detach_capi_ctr(&fbrc_s.ctrl[i]);	
		
	// Release
	fbrc_release_socket(&fbrc_s);		
}

void fbrc_debug(int level, const char *format, ...)
{
    if(level <= debug)
    {
        char output[512];
        va_list args;

        va_start(args, format);
        vsnprintf(output, sizeof(output), format, args);

        printk("<1>%s: %s\n", MNAME, output);
        va_end(args);
    }
}


module_init(fbrc_init);
module_exit(fbrc_exit);

module_param(boost, int, 0);
module_param(debug, int, 0);
module_param(port, int, 0);
module_param_array(controllers, int, &controllers_argc, 0);
module_param_string(addr, addr, sizeof(addr), 0);

MODULE_AUTHOR(MAUTHOR);
MODULE_DESCRIPTION(MNAME " " MVERSION);
MODULE_VERSION(MVERSION);
MODULE_LICENSE("GPL");

