#!/usr/bin/tcl
#Routines for applications to access the wyseman data dictionary
#---------------------------------------
# Copyright (C) 1999-2005 Wyatt-ERP LLC.  All other rights reserved.
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to:
# 
# Free Software Foundation, Inc.
# 51 Franklin Street, Fifth Floor
# Boston, MA  02110-1301, USA

#TODO:
#X- Access a user variable which controls language
#X- cache values from database?
#X- have ::nonull call whole table at once like ::type
#- consolidate ::columns and ::column functions?
#- 
package require wylib
package provide wyseman 0.30

namespace eval wmdd {
    namespace export errtext table type column columns value pkey tables_ref columns_fk
    variable v
    set v(lang) {en}
}

# Return error text
#------------------------------------------------------------
proc wmdd::errtext {table code} {
    variable v
    set idx error:$table:$code
    if {![info exists v($idx)]} {
        set v($idx) [sql::one "select title,help from wm.error_text where tablename = '$table' and code = '$code' and language = '$v(lang)'"]
    }
    return $v($idx)
}

# Return table text
#------------------------------------------------------------
proc wmdd::table {table} {
    variable v
    set idx table:$table
    if {![info exists v($idx)]} {
        set v($idx) [sql::one "select title,help,tablekind from wm.table_pub where tablename = '$table' and language = '$v(lang)'"]
    }
    return $v($idx)
}

# Return name of an oid column (typically _oid) for a view
#------------------------------------------------------------
proc wmdd::view_oid {table} {
    variable v
    set idx view_oid:$table

    if {![info exists v($idx)]} {
        lassign [sql::one "select columnname from wm.column_data where tablename = '$table' and type = 'oid'"] v($idx)
    }
    return $v($idx)
}

# Return tcl boolean indicating if this field can not be null
#------------------------------------------------------------
proc wmdd::nonull {table column} {
    variable v
    set idx nonull:$table:$column

    if {![info exists v($idx)]} {
        set v($idx) [sql::one "select case when nonull then 1 else 0 end from wm.column_data where schemaname = 'public' and tablename = '$table' and columnname = '$column'"]
    }
    return $v($idx)
}

# Return column text
#------------------------------------------------------------
proc wmdd::column {table column} {
    variable v
    set idx column:$table:$column
#puts "column table:$table column:$column idx:$idx"
    if {![info exists v($idx)]} {

# Grab a column at a time
#        set v($idx) [sql::one "select title,help,type from wm.column_pub where tablename = '$table' and columnname = '$column' and language = '$v(lang)'"]

# Grab the whole table at once:
        foreach rec [sql::qlist "select columnname,title,help,type from wm.column_pub where tablename = '$table' and language = '$v(lang)'"] {
            set rec [lassign $rec columnname]
            set ix column:$table:$columnname
            set v($ix) $rec
        }
        if {![info exists v($idx)]} {error "Can't find data for table:$table, column:$column"}
    }
    return $v($idx)
}

# Return column type
#------------------------------------------------------------
proc wmdd::type {table column} {
    return [lindex [column $table $column] 2]
}

# Return all column text for a table
#------------------------------------------------------------
proc wmdd::columns {table} {
    variable v
#puts "wmdd::columns table:$table"
    set idx columns:$table
    if {![info exists v($idx)]} {
        set v($idx) [sql::qlist "select columnname,title,help,type from wm.column_pub where tablename = '$table' and language = '$v(lang)' order by 2"]
    }
    if {$v($idx) == {}} {error "No columns found for $table.  Is the wm.column_native populated?"}
#puts " columns:$v($idx)"
    return $v($idx)
}

# Return allowable values for a column if they exist
#------------------------------------------------------------
proc wmdd::value {table column {val {}}} {
    variable v
    set idx value:$table:$column
#puts "wmdd::value table:$table column:$column"
    if {![info exists v($idx)]} {

#Grab columns one at a time:
#        set v($idx) [sql::qlist "select value,title,help from wm.value_text where tablename = wm.find_native('$table',columnname) and columnname = '$column' and language = '$v(lang)' order by 1"]

#Grab the whole table at once
	foreach rec [columns $table] {
	    set col [lindex $rec 0]
#puts "  set v(value:$table:$col) {}"
	    set v(value:$table:$col) {}
	}
        foreach rec [sql::qlist "select columnname,value,title,help from wm.value_text where tablename = wm.find_native('$table',columnname) and language = '$v(lang)'"] {
            lassign $rec columnname value title help
            lappend v(value:$table:$columnname) [list $value $title $help]
            set v(value:$table:$columnname:$value) [list $value $title $help]
        }
    }
#puts "table:$table column:$column values:$v($idx)"
    if {$val != {}} {set idx "$idx:$val"}
    if {![info exists v($idx)]} {error "No values found for $table:$column.  Is the wm.column_native populated?"}
    if {$val == {}} {return $v($idx)} else {return [lrange $v($idx) 1 end]}
}

# Return the primary key field names for a table
#------------------------------------------------------------
proc wmdd::pkey {table} {
    variable v
    set idx pkey:$table
#puts "wmdd::pkey table:$table"
    if {![info exists v($idx)]} {
        set key {}
        foreach tag [sql::qlist "select columnname from wm.column_pub where tablename = '$table' and pkey order by field"] {
            lappend key $tag
        }
        set v($idx) $key
    }
#puts "  table:$table pkey:$v($idx)"
    return $v($idx)
}

# Find the table,field where a foreign key points to
#------------------------------------------------------------
#proc wmdd::fkey_dest {table column} {
#    set idx fkey_dest:$table:$column
#    if {![info exists v($idx)]} {
#        set v($idx) [sql::one "select ftablename,fcolumns\[1\] from wm.fkeys_pub where tablename = '$table' and columns\[1\] = '$column' limit 1"]
#    }
#    return $v($idx)
#}

# Return tables that are referenced (pointed to) by the specified table
# If refme true, return tables that reference the specified table
#------------------------------------------------------------
proc wmdd::tables_ref {table {refme 0}} {
    variable v
    if {$refme} {
        lassign {tablename ftablename} tcol wcol
    } else {
        lassign {ftablename tablename} tcol wcol
    }

    set ret {}
    set tags {}
    foreach rec [sql::qlist "select 
        k.tablename,k.ftablename,k.key,k.keys,k.colname,k.fcolname,tt.title,tt.help,ct.title,ct.help,k.conname,k.unikey
      from        wm.fkey_pub	k
        left join wm.table_text	tt on tt.tablename = k.${tcol} and tt.language = '$v(lang)' 
        left join wm.column_pub	ct on ct.tablename = k.tablename and ct.columnname = k.ncolname and ct.language = '$v(lang)'
      where k.$wcol = '$table' and has_table_privilege('$table','select') order by k.tablename, k.ftablename, k.key;"] {
        lassign $rec tablename ftablename key keys colname fcolname ttitle thelp ftitle fhelp conname unikey
#puts [format {tablename:%-20s ftablename:%-20s key:%2d keys:%2d colname:%-12s fcolname:%-12s} $tablename $ftablename $key $keys $colname $fcolname]
        set tag "$tablename.$ftablename.$conname.$unikey"
        if {![lcontain $tags $tag]} {lappend tags $tag}		;#store all unique key sets
        lappend cols($tag)	$colname
        lappend fcols($tag)	$fcolname
        lappend ftitles($tag)	$ftitle
        lappend fhelps($tag)	$fhelp
#puts " tag:$tag cols:$cols($tag) fcols:$fcols($tag)"
#puts "   ftitles:$ftitles($tag) fhelps:$fhelps($tag)"
    }
    foreach tag $tags {
        lassign [split $tag .] tablename ftablename conname unikey
        if {[llength $cols($tag)] > 1} {
            lappend ret [list $tablename $cols($tag) $ftablename $fcols($tag) $ttitle $thelp [join $ftitles($tag) {; }] [join $fhelps($tag) {; }]]
        } else {
            lappend ret [list $tablename [lindex $cols($tag) 0] $ftablename [lindex $fcols($tag) 0] $ttitle $thelp [lindex $ftitles($tag) 0] [lindex $fhelps($tag) 0]]
        }
    }
#puts "tables_ref $table $refme ret:$ret"
    return $ret
}

# Return the fk columns in a table and the pk columns they point to in a foreign table
#------------------------------------------------------------
proc wmdd::columns_fk {table ftable} {
    variable v
    set idx columns_fk:$table:$ftable
    if {![info exists v($idx)]} {
#puts "table:$table ftable:$ftable"
        set v($idx) {}
        foreach rec [sql::qlist "select columns,fcolumns from wm.fkeys_pub where tablename = '$table' and ftablename = '$ftable'"] {
            lassign $rec columns fcolumns
#puts " columns:$columns fcolumns:$fcolumns"
            lappend v($idx) [list [eval split $columns ,] [eval split $fcolumns ,]]		;#eval gets rid of {}'s
        }
    }
#puts " v($idx):$v($idx)"
    return $v($idx)
}
