#!/usr/bin/tcl
#Build and maintain a data dictionary for wyseman schemas
#---------------------------------------
# Copyright (C) 1999-2005 Wyatt-ERP LLC.  All other rights reserved.
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to:
# 
# Free Software Foundation, Inc.
# 51 Franklin Street, Fifth Floor
# Boston, MA  02110-1301, USA

#TODO:
#X- Make foreign key references into information_schema
#X- create unified views showing all table/column data
#- get rid of 'short' columns?
#- 
package require wylib
package provide wyseman 0.30

namespace eval wmddict {
    namespace export tabtext schema
    variable v
}

# Handle a structure containing table text information
#------------------------------------------------------------
proc wmddict::tabtext {table args} {
    argform {title help fields} args
    argnorm {{title 2} {short 2} {help 2} {language 2} {fields 1} {errors 2}} args
    array set ca "language en"
    foreach tag {language} {xswitchs $tag args ca($tag)}
    foreach tag {title short help fields errors} {set ca($tag) [sql::esc [xswitchs $tag args]]}
    set    query "delete from wm.table_text where tablename = '$table' and language = '$ca(language)';\n"
    append query "delete from wm.column_text where tablename = '$table' and language = '$ca(language)';\n"
    append query "delete from wm.value_text where tablename = '$table' and language = '$ca(language)';\n"
    append query "delete from wm.error_text where tablename = '$table' and language = '$ca(language)';\n"
    append query "insert into wm.table_text (tablename,language,title,short,help) values ('$table','$ca(language)','$ca(title)','$ca(short)',E'$ca(help)');\n"

    foreach rec $ca(fields) {			;#for each column
        argform {column title help subfields} rec
        argnorm {{column 2} {title 2} {short 2} {help 2} {subfields}} rec
        foreach tag {column title short help subfields} {set cf($tag) [xswitchs $tag rec]}
        append query "insert into wm.column_text (tablename,columnname,language,title,short,help) values ('$table','$cf(column)','$ca(language)','$cf(title)','$cf(short)',E'$cf(help)');\n"

        foreach srec $cf(subfields) {		;#for each subfield
            argform {value title help} srec
            argnorm {{value 2} {title 2} {short 2} {help 2}} srec
            foreach tag {value title short help} {set cs($tag) [xswitchs $tag srec]}
            append query "insert into wm.value_text (tablename,columnname,value,language,title,short,help) values ('$table','$cf(column)','$cs(value)','$ca(language)','$cs(title)','$cs(short)',E'$cs(help)');\n"
        }
    }

    foreach rec $ca(errors) {			;#for each column
        argform {code title help} rec
        argnorm {{code 2} {title 2} {help 2}} rec
        foreach tag {code title help} {set ce($tag) [xswitchs $tag rec]}
        append query "insert into wm.error_text (tablename,code,language,title,help) values ('$table','$ce(code)','$ca(language)','$ce(title)','$ce(help)');\n"

    }
    return $query
}

# Produce SQL to create the data dictionary schema
#------------------------------------------------------------
proc wmddict::schema {} {return $wmddict::schema_text}
set wmddict::schema_text {

    other wm_schema {} {
        create schema wm;
        grant usage on schema wm to public;
    } {
        drop schema wm;
    }

    # Help text for tables
    #-------------------------------------------
    table wm.table_text {wm_schema} {
        tablename	varchar,
        language	varchar not null,
        title		varchar,	-- Normal title
        short		varchar,	-- a very short title		
        help		varchar,	-- longer help description
        primary key (tablename, language)
    } -grant public

    # Help text for columns
    #-------------------------------------------
    table wm.column_text {wm_schema} {
        tablename	varchar,
        columnname	varchar,
        language	varchar not null,
        title		varchar,	-- Normal title
        short		varchar,	-- a very short title		
        help		varchar,	-- longer help description
        primary key (tablename, columnname, language)
    } -grant public
    
    # Help text for enumerated types
    #-------------------------------------------
    table wm.value_text {wm_schema} {
        tablename	varchar,
        columnname	varchar,
        value		varchar,
        language	varchar not null,
        title		varchar,	-- Normal title
        short		varchar,	-- a very short title		
        help		varchar,	-- longer help description
        primary key (tablename, columnname, value, language)
    } -grant public

    # Help text for schema error messages
    #-------------------------------------------
    table wm.error_text {wm_schema} {
        tablename	varchar,
        code		varchar,
        language	varchar not null,
        title		varchar,	-- shorter title for error message
        help		varchar,	-- longer help description
        primary key (tablename, code, language)
    } -grant public
    
    # A table to cache information about the native source(s) of a view's column
    #-------------------------------------------
    table wm.column_native {} {
        tablename	varchar,	-- The name of a view which has some ancestors
        columnname	varchar,	-- The name of a column which comes from those ancestors
        native		varchar,	-- The ancestor which is a table (the ultimate source of column columnname)
        nativecol	varchar,	-- The name of the column in the native table
        pkey		boolean,	-- Frontend tools should consider this column as part of the primary key
        primary key (tablename, columnname)	-- each column can have only zero or one table considered as its native source
    } -grant public
    index {} wm.column_native native

    # The rest is an abstraction layer on postgres system tables and the tables 
    # above to create a data dictionary describing our schema
    #-------------------------------------------

    # Backend information about tables
    #-------------------------------------------
    view wm.table_data {wm_schema} {
        select
        ns.nspname	as "schemaname", 
        cl.relname	as "tablename", 
        cl.relkind	as "tablekind",
        cl.relhaspkey	as "haspkey",
        cl.relnatts	as "columns"
        from pg_class cl join pg_namespace ns on cl.relnamespace = ns.oid
        where cl.relkind in ('r','v');	--only show tables and views
    } -grant public

    # A version of the similar view from information_schema, but as of now, it
    # is much faster (probably helps to avoid the pg_user lookup...)
    #-------------------------------------------
    view wm.view_column_usage {wm_schema} {
        select
        nv.nspname	as "view_schema",
        v.relname	as "view_name",   
        nt.nspname	as "table_schema",
        t.relname	as "table_name",   
--        t.relkind	as "table_kind",
        at.attname	as "column_name",
        t.relowner	as "owner"
    from	pg_depend	dv
        join	pg_class	v	on v.oid         = dv.refobjid
        join 	pg_namespace	nv	on nv.oid        = v.relnamespace
        join	pg_depend	dt	on dt.objid      = dv.objid
                                       and dt.refobjid  <> dv.refobjid
                                       and dt.classid    = dv.classid    
                                       and dt.refclassid = dv.refclassid 
        join	pg_class	t	on t.oid         = dt.refobjid
        join	pg_namespace	nt	on nt.oid        = t.relnamespace
        join	pg_attribute	at	on at.attrelid   = dt.refobjid and at.attnum = dt.refobjsubid

    where	dv.deptype = 'i'
        and	v.relkind = 'v'
        and	t.relkind IN ('r', 'v')
        and	dv.classid    = 'pg_catalog.pg_rewrite'::regclass
        and	dv.refclassid = 'pg_catalog.pg_class'::regclass

        and	nv.nspname = 'public'
        and	nt.nspname = 'public';
    } -grant public

    # Given a table and column name, return the name of the table/view in which
    # this column is natively found.  For example, if this is a view and the 
    # column refers to a column in an underlying table, return the name of that 
    # table.  If the column was manufactured in this view, return the name of 
    # the view.  Go back any number of generations to find the table/view in 
    # which the column is native.
    # We will use this to fill up the table wm.column_native (so its faster)
    #-------------------------------------------
    function {wm.find_native(tab varchar, col varchar)} {wm_schema wm.view_column_usage} {
      returns varchar language plpgsql stable security definer as $$
        declare
            rec		record;
        begin
            -- There can be more than one table from which a column derives.  We will take the first one (the schema author can override this with the -native switch).
            select into rec * from wm.view_column_usage where view_schema = 'public' and view_name = tab and column_name = col order by table_name desc limit 1;
            if not found then		-- column not found in view table
                return tab;
            end if;
            return wm.find_native(rec.table_name::varchar, col);
        end;
      $$;
    }

    #Backend information about columns
    #-------------------------------------------
    view wm.column_data {wm_schema wm.column_native} {
      select
        n.nspname	as "schemaname"
      , c.relname	as "tablename"
      , a.attnum	as "field"
      , a.attname	as "columnname"
      , t.typname	as "type"
      , na.attnotnull	as "nonull"		-- notnull of native table
      , case when a.attlen < 0 then null else a.attlen end 	as "length"
      , coalesce(na.attnum = any((select conkey from pg_constraint
            where connamespace = c.relnamespace
            and conrelid = nc.oid and contype = 'p')::int4[]),'f') as "ispkey"
      , ts.pkey			-- like ispkey, but can be overridden explicitly in the wms file
      , ts.native		-- much faster than calling wm.find_native(), but the values must be present in the summary table wm.column_native
      , ts.nativecol
      from		pg_class	c
          join		pg_attribute	a	on a.attrelid =	c.oid
          join		pg_type		t	on t.oid = a.atttypid
          join		pg_namespace	n	on n.oid = c.relnamespace
          left join	wm.column_native ts	on ts.tablename = c.relname and ts.columnname = a.attname
          left join	pg_class	nc	on nc.relnamespace = c.relnamespace and nc.relname = ts.native
          left join	pg_attribute	na	on na.attrelid = nc.oid and na.attname = a.attname
      where a.attnum >= 0 			-- don't include system columns
        and c.relkind in ('r','v');		-- only include tables and views
    } -grant public

    #Unified information about tables in the public schema
    #-------------------------------------------
    view wm.table_pub {wm.table_data wm.table_text} {
        select
        td.tablename, td.tablekind, td.haspkey, td.columns,
        tt.language,
        tt.title,
        tt.short,
        tt.help
        from (wm.table_data td left join wm.table_text tt on td.tablename = tt.tablename)
        where td.schemaname = 'public';
    } -grant public

    #Unified information about table columns in the public schema
    #-------------------------------------------
    view wm.column_pub {wm.column_data wm.column_text} {
      select
        cd.tablename
      , cd.field
      , cd.columnname
      , cd.type
      , cd.nonull
      , cd.length
      , cd.ispkey
      , cd.pkey
      , cd.native
      , cd.nativecol
      , coalesce(vt.language,nt.title,'en')		as language
      , coalesce(vt.title,nt.title,cd.columnname)	as title
      , coalesce(vt.short,nt.short)			as short
      , coalesce(vt.help, nt.help)			as help
      from		wm.column_data cd
        left join	wm.column_text vt	on vt.tablename = cd.tablename and vt.columnname = cd.columnname
        left join	wm.column_text nt	on nt.tablename = cd.native    and nt.columnname = cd.nativecol

      where cd.schemaname  = 'public'
        and cd.columnname != '_oid'
    } -grant public

    #Generate an array of column names from their position numbers
    #-------------------------------------------
    function {wm.column_names(oid,int4[])} {wm_schema plpgsql} {
      returns varchar[] as $$
        declare
            val	varchar[];
            rec	record;
        begin
-- Would be nice if something like this worked:
--          select into val attname from pg_attribute where attrelid = $1 and attnum = any($2);
    
            for rec in select * from pg_attribute where attrelid = $1 and attnum = any($2) loop
                if val isnull then
                    val := array[rec.attname::varchar];
                else
                    val := val || rec.attname::varchar;
                end if;
            end loop;
            return val;
        end;
      $$ language plpgsql stable;
    }

    #information about primary/foreign keys
    #-------------------------------------------
    view wm.fkeys_data {wm_schema wm.column_names(oid,int4[])} {
      select
        ns.nspname	as "schemaname",
        tc.relname	as "tablename",
        co.contype	as "type",
        co.conname	as "conname",
        co.conkey	as "fields",
        wm.column_names(co.conrelid,co.conkey) as "columns",
        fc.relname	as "ftablename",
        co.confkey	as "ffields",
        wm.column_names(co.confrelid,co.confkey) as "fcolumns"
      from		pg_constraint	co 
        join		pg_namespace	ns on co.connamespace = ns.oid
        join		pg_class	tc on co.conrelid = tc.oid
        left join	pg_class	fc on co.confrelid = fc.oid
      where co.contype in ('f','p');
    } -grant public

    #Information about foreign keys to public tables/views
    #-------------------------------------------
    view wm.fkeys_pub {wm_schema wm.fkeys_data wm.column_native} {
      select
        tn.tablename,tn.native,tk.columns,
        fn.tablename as "ftablename",fn.native as "fnative", tk.fcolumns

      from	wm.fkeys_data		tk
        join	wm.column_native	tn on tn.native = tk.tablename  and tn.nativecol = tk.columns[1]
        join	wm.column_native	fn on fn.native = tk.ftablename and fn.nativecol = tk.fcolumns[1]

      where tk.schemaname = 'public' and tk.type = 'f';
    } -grant public
    
    #information about primary/foreign keys
    #-------------------------------------------
    view wm.fkey_data {wm_schema} {
      select
        ns.nspname	as "schemaname",
        tc.relname	as "tablename",
        co.contype	as "type",
        co.conname	as "conname",
        co.conkey	as "fields",
        co.conkey[s.a]	as "field",
        ta.attname	as "colname",
        fc.relname	as "ftablename",
        co.confkey	as "ffields",
        co.confkey[s.a]	as "ffield",
        fa.attname	as "fcolname",
        s.a		as "key",
        array_upper(co.conkey,1)	as "keys"
      from		pg_constraint	co 
        join		generate_series(1,10) s(a)	on true
        join		pg_attribute	ta on ta.attrelid = co.conrelid  and ta.attnum = co.conkey[s.a]
        join		pg_attribute	fa on fa.attrelid = co.confrelid and fa.attnum = co.confkey[s.a]
        join		pg_namespace	ns on co.connamespace = ns.oid
        join		pg_class	tc on co.conrelid = tc.oid
        left join	pg_class	fc on co.confrelid = fc.oid
      where co.contype = 'f';
    } -grant public
    
    #Information about foreign keys to public tables/views
    #-------------------------------------------
    view wm.fkey_pub {wm_schema wm.fkey_data wm.column_native} {
      select
        kd.conname
      , kd.key
      , kd.keys
      , tn.tablename	as "tablename"
      , fn.tablename	as "ftablename"
      , tn.columnname	as "colname"
      , kd.colname	as "ncolname"
      , fn.columnname	as "fcolname"
      , tn.native	as "native"
      , fn.native	as "fnative"

      , case when exists (select * from wm.column_native where tablename = tn.tablename and native = tn.native and columnname != tn.columnname and nativecol = kd.colname) then
            tn.columnname
        else
            null
        end as "unikey"		-- used to differentiate between multiple fkeys pointing to the same destination, and multi-field fkeys pointing to multi-field destinations
      , coalesce(vt.language,nt.title,'en')		as language
      , coalesce(vt.title,nt.title,tn.columnname)	as title
      , coalesce(vt.short,nt.short)			as short
      , coalesce(vt.help, nt.help)			as help

      from	wm.fkey_data		kd
        join	wm.column_native	tn on tn.native = kd.tablename  and tn.nativecol = kd.colname
        join	wm.column_native	fn on fn.native = kd.ftablename and fn.nativecol = kd.fcolname
        left join wm.column_text vt	on vt.tablename = tn.tablename and vt.columnname = tn.columnname
        left join wm.column_text nt	on nt.tablename = tn.native    and nt.columnname = kd.colname

      where kd.schemaname = 'public';
    } -grant public
    
    #Standard plpgsql lanaguage handler
    #-------------------------------------------
    function plpgsql_call_handler() {} {
        returns language_handler as '/usr/local/pgsql/lib/plpgsql.so' language 'C';
    }

    #Creates the plpgsql language
    #-------------------------------------------
    other plpgsql plpgsql_call_handler() {
        create trusted language 'plpgsql' 
        handler plpgsql_call_handler lancompiler 'PL/pgSQL';
    } {drop language plpgsql;}

    #PLTCL Language handler
    #-------------------------------------------
    function pltcl_call_handler() {} {
        returns language_handler as '/usr/local/pgsql/lib/pltcl.so' language 'C';
    }

    #Creates the pltcl language
    #-------------------------------------------
    other pltcl pltcl_call_handler() {
        create trusted language 'pltcl'
        handler pltcl_call_handler lancompiler 'PL/TCL';
    } {drop language pltcl;}    
}

# Blocks of initialization code to be executed in order after creating/modifying tables/views in schema
#------------------------------------------------------------
proc wmddict::init_sql {{idx 0}} {

    if {$idx <= 0} {
        #wm.column_native needs to know the native table for all view columns
        set sql "delete from wm.column_native;\n"
        append sql "insert into wm.column_native (tablename, columnname, native, nativecol, pkey) select tablename, columnname, wm.find_native(tablename::varchar, columnname::varchar), columnname, 'f' from wm.column_data where schemaname = 'public' and columnname != '_oid';\n"
    
        foreach natrec [wmparse::natives] {		;#update all forced native fields
    #puts "natrec:$natrec"
            foreach col [lassign $natrec tab nat] {
                if {[llength $col] > 1} {lassign $col col ncol} else {set ncol $col}
                append sql "update wm.column_native set native = '$nat', nativecol = '$ncol' where tablename = '$tab' and columnname = '$col';\n"
            }
        }
    } elseif {$idx == 1} {

        foreach rec [sql::qlist "select tablename,columnname from wm.column_data where ispkey and schemaname = 'public' and columnname != '_oid' order by 1,2"] {		;#ispkey field was not yet valid in first insert so is default false
            lassign $rec tab col
            append sql "update wm.column_native set pkey = 't' where tablename = '$tab' and columnname = '$col';\n"
        }

        foreach prirec [wmparse::primaries] {		;#update all forced primary keys
#puts "prirec:$prirec"
             lassign $prirec tab pkcols
             append sql "update wm.column_native set pkey = 'f' where tablename = '$tab' and not columnname in ('[join $pkcols {','}]');\n"
             append sql "update wm.column_native set pkey = 't' where tablename = '$tab' and     columnname in ('[join $pkcols {','}]');\n"
        }
    } else {
        set sql {}
    }

#puts "sql:$sql"
    return $sql
}
