package provide wylib 0.30
package require http
package require tls

http::register https 443 [list ::tls::socket -require 1 -cafile /etc/pki/tls/certs/ca-bundle.crt]

namespace eval fedex {
    namespace export carrierCode packagesForMethod listMethods expressAvail ratePackage ratePackageAll shipPackage cancelPackage closeShipments trackPackage
    variable cnf
    # these defaults are non-usable. please set them before using
    set cnf(account) 000000000
    set cnf(meter) 0000000
    set cnf(host) https://gatewaybeta.fedex.com:443/GatewayDC
    set cnf(sargs) {{saddress {}} {scity {}} {sstate {}} {szip {}} {emails {}} {shipdate {}} {heights {}} {widths {}} {lengths {}} {packages 1} {packaging 01} {residential N} {satpickup {}} {satdelivery {}} {signature 2} {comm_info {}} {printer {}}}
    set cnf(comp) {}
    set cnf(fax) {}
    set cnf(phone) {}
}


#--- Format a floating point number to a given number of decimal places
proc fedex::float {num places} {
    foreach {int frac} [split $num "."] break
    if {[string length $frac] >= $places} {
        set frac [string range $frac 0 [expr {$places-1}]]
    } else {
        append frac [string repeat 0 [expr {$places - [string length $frac]}]]
    }
    return "$int\.$frac"
}

#--- returns the carrier code
# express is either 1 for express, or 0 for standard
# return FDXE or FDXG
proc fedex::carrierCode {express} {
    if {$express} {return FDXE}
    return FDXG
}

#--- return a list of package types (for FedEx) for a given service type and package type (from the database)
# express is either 1 for express, or 0 for standard
# type is one of box, let, pak, or tub
# optionally takes international (-int 1)
# returns as list of code / name
# on error, returns a blank list
proc fedex::packagesForMethod {express type args} {
    # parse optional arguments
    argform {international} args
    argnorm {{international 1 int}} args
    argproc \$s $args {{int 0}}

    if {$express} {
        switch $type {
            box {
                if {$int} {
                    return [list {01 {Customer Packaging}} {03 {FedEx Box}} {15 {FedEx 10kg Box}} {25 {FedEx 25kg Box}}]
                } else {
                    return [list {01 {Customer Packaging}} {03 {FedEx Box}}]
                }
            }
            let {return [list {06 {FedEx Envelope}}]}
            pak {return [list {02 {FedEx Pak}}]}
            tub {return [list {04 {FedEx Tube}}]}
            default {return [list]}
        }
    } else {
        switch $type {
            box {return [list {01 {Customer Packaging}}]}
            default {return [list]}
        }
    }
}

#--- return a list of available service types (excluding freight)
# optionally takes international (-int 1)
# returns as list of express (boolean) / code / name
proc fedex::listMethods {args} {
    # parse optional arguments
    argform {international} args
    argnorm {{international 1 int}} args
    argproc \$s $args {{int 0}}

    if {$int} {
        return [list {1 01 {FedEx International Priority}} {1 03 {FedEx International Economy}} {1 06 {FedEx International First}} {0 92 {FedEx International Ground Service}}]
    } else {
        return [list {1 01 {FedEx Priority}} {1 03 {FedEx 2day}} {1 05 {FedEx Standard Overnight}} {1 06 {FedEx First Overnight}} {1 20 {FedEx Express Saver}} {0 90 {FedEx Home Delivery}} {0 92 {FedEx Ground Service}}]
    }
}

#--- open a connection to fedex server (through the ssl pipe on tao) and return the result
# takes raw input, returns raw output
proc fedex::send_receive {send} {
    variable cnf
    set tok [http::geturl $cnf(host) -query $send]
    upvar #0 $tok state
    set receive $state(body)
    unset state
    http::cleanup $tok
    return $receive
}

#--- decode the received text (receive) from fedex into an existing array (arr)
# turns 0,"211"99,"" into arr(0) = 211, arr(99) = {}
# takes raw input and array name, no return value
proc fedex::decode {receive arr} {
    upvar $arr keys
     
    while {[string length $receive] > 0} {
        if {[regexp {[-0-9]+,\"[^\"]*\"} $receive part] == 0} {set receive {}}
        if {$receive == {}} { return }
        regexp {([-0-9]+),\"([^\"]*)\"} $part match key value
        set keys($key) $value
        set receive [string range $receive [string length $part] [string length $receive]]
    }
}

#--- encode a list of pairs to be suitable for sending to fedex
# turns {{0 211} {99 {}}} into 0,"211"99,""
# takes formatted input, returns raw input
proc fedex::encode {send} {
    set retval {}
    foreach i $send {
        append retval [lindex $i 0],"[lindex $i 1]"
    }
    return $retval
}

#--- Checks fedex express service availability for the given parameters
# requires recipient zip, recipient country code
# also takes sender zip (84601), ship date (today), number of packages (1)
# returns a list of available delivery dates, destination station ID, and service type
# ship date is in YYYYMMDD format {%Y%m%d}
# on error, returns the error message
proc fedex::expressAvail {rzip rcc args} {
    variable cnf

    # parse optional arguments
    argform {szip shipdate packages} args
    argnorm {{szip 2} {shipdate 5} {packages 7}} args
    argproc \$s $args {{szip 84601} {shipdate {}} {packages 1}}

    if {$shipdate == {}} { set shipdate [clock format [clock seconds] -format {%Y%m%d}] }

    # build the input data
    set send {}
    set s lappend
    $s send "0 019"
    $s send "10 $cnf(account)"
    $s send "498 $cnf(meter)"
    $s send "3025 FDXE"
    $s send "9 $szip"
    $s send "117 US"
    $s send "17 $rzip"
    $s send "50 $rcc"
    $s send "24 $shipdate"
    $s send "116 $packages"
    $s send {99 {}}

    # send and receive
    set receive [send_receive [fedex::encode $send] ]
    set array ret
    fedex::decode $receive ret

    # -- parse decoded reply

    # none avail or errored out?
    if {[info exists ret(2)] == 1} {
        return $ret(3)
    }

    set retval {}
    loop var 0 [string trimleft $ret(1133) 0] 1 {
        set i [expr {$var+1}]
        set val {}
        if {[info exists ret(409-$i)] == 1} {
            lappend val $ret(409-$i)
        } else {
            lappend val {}
        }
        if {[info exists ret(195-$i)] == 1} {
            lappend val $ret(195-$i)
        } else {
            lappend val {}
        }
        if { [info exists ret(1274-$i)] == 1} {
            lappend val $ret(1274-$i)
        } else {
            lappend val {}
        }

        lappend retval $val
    }

    return $retval
}

#--- Gets a rate quote for a package, when the service type is known (ground or express)
# requires recipient state, recipient zip, recipient country code, package weight, declared value, carrier, service type
# also takes sender state, sender zip, ship date (today), package dimensions (blank), number of packages (1), packaging type (customer), residential delivery (no), saturday pickup (blank), saturday delivery (blank), delivery signature type (indirect)
# returns the USD quote, with two decimal places
# weight is in pounds, requires 1 decimal place
# value is in USD and must have two decimal places trailing the dollar amount. no dollar sign
# carrier is either FDXE for express or FDXG for ground
# service is one of the following:
#  FedEx Express U.S. Domestic shipping service types are:
#   01 - FedEx Priority
#   03 - FedEx 2day
#   05 - FedEx Standard Overnight
#   06 - FedEx First Overnight
#   20 - FedEx Express Saver
#   70 - FedEx 1day Freight
#   80 - FedEx 2day Freight
#   83 - FedEx 3dayFreight
#  FedEx Express International shipping service types are:
#   01 - FedEx International Priority
#   03 - FedEx International Economy
#   06 - FedEx International First
#   70 - FedEx International Priority Freight
#   86 - FedEx International Economy Freight
#  FedEx Ground U.S. Domestic shipping service types are:
#   90 - FedEx Home Delivery
#   92 - FedEx Ground Service
#  FedEx Ground International shipping service types are:
#   92 - FedEx Ground Service
# ship date is in YYYYMMDD format {%Y%m%d}
# height, width, and length are in inches
# packaging is one of the following: (must be 01 when carrier is FDXG)
#   01: customer packaging
#   02: FedEx Pak
#   03: FedEx Box
#   04: FedEx Tube
#   06: FedEx Envelope
#   15: FedEx 10kg box (international only)
#   25: FedEx 25kg box (international only)
# residential delivery can either be N or Y
# saturday pickup and delivery can either be blank, N, or Y
# signature type is one of the following:
#   1: deliver without signature
#   2: indirect signature (recipient, neighbor, or through a release)
#   3: direct signature (recipient only)
#   4: adult signature (21 and over recipient)
# on error, returns the error message
proc fedex::ratePackage {rstate rzip rcc weight value carrier service args} {
    variable cnf

    # parse optional arguments
    argform {sstate szip shipdate height width length packages packaging residential satpickup satdelivery signature} args
    argnorm {{sstate 2} {szip 2} {shipdate 5} {height 1} {width 1} {length 1} {packages 7} {packaging 7} {residential 1} {satpickup 4} {satdelivery 4} {signature 2}} args
    argproc \$s $args {{sstate UT} {szip 84601} {shipdate {}} {height {}} {width {}} {length {}} {packages 1} {packaging 01} {residential N} {satpickup {}} {satdelivery {}} {signature 2}}

    if {$shipdate == {}} { set shipdate [clock format [clock seconds] -format {%Y%m%d}] }

    # build the input data
    set send {}
    set s lappend
    $s send "0 022"
    $s send "10 $cnf(account)"
    $s send "498 $cnf(meter)"
    $s send "3025 $carrier"
    $s send "8 $sstate"
    $s send "9 $szip"
    $s send "117 US"
    $s send "16 $rstate"
    $s send "17 $rzip"
    $s send "50 $rcc"
    $s send "1529 2"
    $s send "24 $shipdate"

    if {$height != {} && $width != {} && $length != {}} {
        $s send "57 $height"
        $s send "58 $width"
        $s send "59 $length"
    }

    $s send "68 USD"
    $s send "75 LBS"
    $s send "116 $packages"

    if {$height != {} && $width != {} && $length != {}} {
        $s send "1116 I"
    }

    $s send "1273 $packaging"
    $s send "1274 $service"
    $s send "1401 [fedex::float $weight 1]"
    $s send "1415 [fedex::float $value 2]"
    $s send "440 $residential"
    $s send "1266 $satdelivery"
    $s send "1267 $satpickup"
    $s send "1333 1"
    $s send "2399 $signature"
    $s send {99 {}}

    # send and receive
    set receive [send_receive [fedex::encode $send] ]
    set array ret
    fedex::decode $receive ret
    
    # -- parse decoded reply

    # none avail or errored out?
    if {[info exists ret(2)] == 1} {
        return $ret(3)
    }

    if {[info exists ret(1416)] == 1 } {
        return $ret(1416)
    }

    return {}
}

#--- Gets a list of rate quotes for a package, for all available services
# requires recipient state, recipient zip, recipient country code, package weight, declared value
# takes sender state, sender zip, carrier (all), ship date (today), package dimensions (blank), number of packages (1), packaging type (customer), residential delivery (no), saturday pickup (blank), saturday delivery (blank), delivery signature type (indirect)
# returns list of rate quotes, service types, and estimated transit times (for FDXG only)
# weight is in pounds, requires 1 decimal place
# value is in USD and must have two decimal places trailing the dollar amount. no dollar sign
# carrier is either FDXE for express or FDXG for ground
# ship date is in YYYYMMDD format {%Y%m%d}
# height, width, and length are in inches
# packaging is one of the following: (must be 01 when carrier is FDXG)
#   01: customer packaging
#   02: FedEx Pak
#   03: FedEx Box
#   04: FedEx Tube
#   06: FedEx Envelope
#   15: FedEx 10kg box (international only)
#   25: FedEx 25kg box (international only)
# residential delivery can either be N or Y
# saturday pickup and delivery can either be blank, N, or Y
# signature type is one of the following:
#   1: deliver without signature
#   2: indirect signature (recipient, neighbor, or through a release)
#   3: direct signature (recipient only)
#   4: adult signature (21 and over recipient)
# on error, returns the error message
proc fedex::ratePackageAll {rstate rzip rcc weight value args} {
    variable cnf

    # parse optional arguments
    argform {sstate szip carrier shipdate height width length packages packaging residential satpickup satdelivery signature} args
    argnorm {{sstate 2} {szip 2} {carrier 1} {shipdate 5} {height 1} {width 1} {length 1} {packages 7} {packaging 7} {residential 1} {satpickup 4} {satdelivery 4} {signature 2}} args
    argproc \$s $args {{sstate UT} {szip 84601} {carrier {}} {shipdate {}} {height {}} {width {}} {length {}} {packages 1} {packaging 01} {residential N} {satpickup {}} {satdelivery {}} {signature 2}}

    if {$shipdate == {}} { set shipdate [clock format [clock seconds] -format {%Y%m%d}] }

    # build the input data
    set send {}
    set s lappend
    $s send "0 025"
    $s send "10 $cnf(account)"
    $s send "498 $cnf(meter)"

    if {$carrier != {}} {
        $s send "3025 $carrier"
    }
    $s send "8 $sstate"
    $s send "9 $szip"
    $s send "117 US"
    $s send "16 $rstate"
    $s send "17 $rzip"
    $s send "50 $rcc"
    $s send "1529 2"
    $s send "24 $shipdate"

    if {$height != {} && $width != {} && $length != {}} {
        $s send "57 $height"
        $s send "58 $width"
        $s send "59 $length"
    }

    $s send "68 USD"
    $s send "75 LBS"
    $s send "116 $packages"

    if {$height != {} && $width != {} && $length != {}} {
        $s send "1116 I"
    }

    $s send "1273 $packaging"
    $s send "1401 [fedex::float $weight 1]"
    $s send "1415 [fedex::float $value 2]"
    $s send "440 $residential"
    $s send "1266 $satdelivery"
    $s send "1267 $satpickup"
    $s send "1333 1"
    $s send "2399 $signature"
    $s send {99 {}}

    # send and receive
    set receive [send_receive [fedex::encode $send] ]
    set array ret
    fedex::decode $receive ret
    
    # -- parse decoded reply

    # none avail or errored out?
    if {[info exists ret(2)] == 1} {
        return $ret(3)
    }

    # "soft" error?
    if {[info exists ret(557)] == 1} {
        return $ret(559)
    }

    set retval {}
    loop var 0 [string trimleft $ret(1133) 0] 1 {
        set i [expr {$var+1}]
        set val {}
        lappend val $ret(1416-$i)
        if {[info exists ret(1274-$i)] == 1} {
            lappend val $ret(1274-$i)
        } else {
            lappend val {}
        }
        if {[info exists ret(3058-$i)] == 1} {
            lappend val $ret(3058-$i)
        } else {
            lappend val {}
        }

        lappend retval $val
    }
    return $retval
}

#--- Ships a package and prints a shipping label
# requires recipient company name, contact name, address1, address2, state, zip, phone, and country code, package weight list, declared value, carrier, service type
# also takes sender address, state, and, zip, ship date (today), package dimensions lists (blank), number of packages (1), packaging type (customer) list, residential delivery (no), saturday pickup (blank), saturday delivery (blank), delivery signature type (indirect), commodity information (blank), and printer (TP1)
# recipient phone is 10 numbers only
# weight is in pounds, requires 1 decimal place, either singly or in a list
# value is in USD and must have two decimal places trailing the dollar amount. no dollar sign
# carrier is either FDXE for express or FDXG for ground
# service is one of the following:
#  FedEx Express U.S. Domestic shipping service types are:
#   01 - FedEx Priority
#   03 - FedEx 2day
#   05 - FedEx Standard Overnight
#   06 - FedEx First Overnight
#   20 - FedEx Express Saver
#   70 - FedEx 1day Freight
#   80 - FedEx 2day Freight
#   83 - FedEx 3dayFreight
#  FedEx Express International shipping service types are:
#   01 - FedEx International Priority
#   03 - FedEx International Economy
#   06 - FedEx International First
#   70 - FedEx International Priority Freight
#   86 - FedEx International Economy Freight
#  FedEx Ground U.S. Domestic shipping service types are:
#   90 - FedEx Home Delivery
#   92 - FedEx Ground Service
#  FedEx Ground International shipping service types are:
#   92 - FedEx Ground Service
# ship date is in YYYYMMDD format {%Y%m%d}
# height, width, and length are in inches, either singly or in a list
# packaging is one of the following: (must be 01 when carrier is FDXG) either singly or as a list
#   01: customer packaging
#   02: FedEx Pak
#   03: FedEx Box
#   04: FedEx Tube
#   06: FedEx Envelope
#   15: FedEx 10kg box (international only)
#   25: FedEx 25kg box (international only)
# residential delivery can either be N or Y
# saturday pickup and delivery can either be blank, N, or Y
# signature type is one of the following:
#   1: deliver without signature (not available for FDXE)
#   2: indirect signature (recipient, neighbor, or through a release)
#   3: direct signature (recipient only)
#   4: adult signature (21 and over recipient)
# comm_info is for customs. It needs to be a list of the following list (up to 20):
#   number of pieces
#   description (not vague)
#   weight per unit
#   value per unit
# returns the tracking number, form ID, master tracking number, and master form ID for each package shipped
# on error, returns the error message
proc fedex::shipPackage {rcomp rname raddress1 raddress2 rcity rstate rzip rphone rcc weights value carrier service args} {
    variable cnf

    # parse optional arguments
    argform {saddress scity sstate szip emails shipdate height width length packages packaging residential satpickup satdelivery signature comm_info printer} args
    argnorm {{saddress 2} {scity 2} {sstate 2} {szip 2} {emails 2} {shipdate 5} {heights 1} {widths 1} {lengths 1} {packages 7} {packaging 7} {residential 1} {satpickup 4} {satdelivery 4} {signature 2} {comm_info 1} {printer 1}} args
    argproc \$s $args $cnf(sargs)

    if {$shipdate == {}} { set shipdate [clock format [clock seconds] -format {%Y%m%d}] }

    set retval {}
    set master_track {}
    set master_form {}
    set tweight 0
    foreach w $weights {
        set tweight [expr $tweight + $w]
    }

    # run transaction once for each item in the package
    loop packnum 0 $packages 1 {

        # build the input data
        set send {}
        set s lappend
        $s send "0 021"
        $s send "10 $cnf(account)"
        $s send "498 $cnf(meter)"
        $s send "3025 $carrier"
        $s send [list 4 $cnf(comp)]
        $s send [list 5 "$saddress"]
        $s send [list 7 "$scity"]
        $s send "8 $sstate"
        $s send "9 $szip"
        $s send "117 US"
        $s send [list 183 $cnf(phone)]
        $s send [list 1103 $cnf(fax)]
        $s send [list 11 "$rcomp"]
        $s send [list 12 "$rname"]
        $s send [list 13 "$raddress1"]
        if {$raddress2 != {}} {
            $s send [list 14 "$raddress2"]
        }
        $s send [list 15 "$rcity"]
        $s send "16 $rstate"
        $s send "17 $rzip"
        $s send "18 $rphone"
        $s send "50 $rcc"
    
        if {$emails != {}} {
            $s send "1202 [lindex $emails 0]"
        }

        $s send "24 $shipdate"
        $s send [list 1115 [clock format [clock seconds] -format {%H%M%S}]]
        $s send "1119 Y"
        $s send "1529 2"
        $s send "23 1"
        $s send "68 USD"
        $s send "70 1"
        $s send "116 $packages"

        if {$packages > 1} {
            $s send "1117 [expr $packnum + 1]"
            
            if {$packnum >= 1} {
                $s send "1123 $master_track"

                if {$master_form != {}} {
                    $s send "1124 $master_form"
                }
            }
        }

        if {$packnum == 0} {
            $s send "1400 [fedex::float $tweight 1]"
        }

        if {$heights != {} && $widths != {} && $lengths != {}} {
            $s send "57 [lindex $heights $packnum]"
            $s send "58 [lindex $widths $packnum]"
            $s send "59 [lindex $lengths $packnum]"
        }

        $s send "73 N"
        $s send "75 LBS"
        $s send "1116 I"
        $s send "1273 [lindex $packaging $packnum]"
        $s send "1274 $service"
        $s send "1401 [fedex::float [lindex $weights $packnum] 1]"
        $s send "1411 [fedex::float $value 2]"
        $s send "1415 [fedex::float $value 2]"
        
        loop i 0 [llength $emails] 1 {
            $s send "1204-$i [lindex $emails $i]"
        }

        $s send "440 $residential"
        $s send "1266 $satdelivery"
        $s send "1267 $satpickup"
        $s send "1333 1"
        $s send "2399 $signature"
        $s send "1368 2"
        $s send "1369 2"
        $s send "1370 4"

        loop comm 0 [llength $comm_info] 1 {
            set i [expr {$comm + 1}]
            $s send "76-$i [lindex [lindex $comm_info $comm] 0]"
            $s send [list 79-$i "[lindex [lindex $comm_info $comm] 1]"]
            $s send "80-$i US"
            $s send "82-$i [lindex [lindex $comm_info $comm] 0]"
            $s send "414-$i LBS"
            $s send "1407-$i [lindex [lindex $comm_info $comm] 2]"
            $s send "1408-$i [fedex::float [lindex [lindex $comm_info $comm] 3] 1]"
            $s send "1410-$i [fedex::float [expr {[lindex [lindex $comm_info $comm] 3] * [lindex [lindex $comm_info $comm] 0]}] 6]"
        }

        $s send {99 {}}

        # send and receive
#puts $send
        set receive [send_receive [fedex::encode $send] ]
        set array ret
        fedex::decode $receive ret
#puts $receive
    
        # -- parse decoded reply

        # none avail or errored out?
        if {[info exists ret(2)] == 1} {
            return $ret(3)
        }

        # "soft" error?
        if {[info exists ret(557)] == 1} {
            return $ret(559)
        }

        set val {}
        lappend val $ret(29)
        if {[info exists ret(526)] == 1} {
            lappend val $ret(526)
        } else {
            lappend val {}
        }
        if {[info exists ret(1123)] == 1} {
            lappend val $ret(1123)
            set master_track $ret(1123)
        } else {
            lappend val {}
        }
        if {[info exists ret(1124)] == 1} {
            lappend val $ret(1124)
            set master_form $ret(1124)
        } else {
            lappend val {}
        }
        if {[info exists ret(1419)] == 1} {
            lappend val $ret(1419)
        } else {
            lappend val {}
        }
        if {[info exists ret(188)] == 1} {
            set buffer $ret(188)
            regsub -all "\%00" $buffer [binary format a {}] buffer1
            regsub -all "\%22" $buffer1 \" buffer
            regsub -all "\%25" $buffer % buffer1
            set o [open "|lpr -P$printer -oraw" w]
            fconfigure $o -translation binary
            puts -nonewline $o $buffer1
            close $o
        }
        
        lappend retval $val
    }

    return $retval
}

#--- Cancel a package before it's sent
# requires the carrier and tracking number
# returns nothing on success
# carrier is either FDXE for express or FDXG for ground. must be the same as sent to shipPackage
# tracking must be the same tracking number given from shipPackage
# on error, returns the error message
proc fedex::cancelPackage {carrier tracking} {
    variable cnf

    # build the input data
    set send {}
    set s lappend
    $s send "0 023"
    $s send "10 $cnf(account)"
    $s send "498 $cnf(meter)"
    $s send "3025 $carrier"
    $s send "29 $tracking"
    $s send {99 {}}

    # send and receive
    set receive [send_receive [fedex::encode $send] ]
    set array ret
    fedex::decode $receive ret
    
    # -- parse decoded reply

    # none avail or errored out?
    if {[info exists ret(2)] == 1} {
        return $ret(3)
    }

    return {}
}


#--- Close out previous ground shipments made this day. After this is run, no previous shipments can be cancelled. Shipments made after the close can still be cancelled. Closing is not required to ship.
# requires no arguments
# optionally takes close date/time (today/now) and retrieve previous manifest (N)
# returns the shipping manifest on success
# close date is in YYYYMMDD format {%Y%m%d}
# close time is in HHMMSS format {%H%M%S}
# current is either N or Y
# on error, returns the error message
proc fedex::closeShipments {args} {
    variable cnf

    # parse optional arguments
    argform {closedate closetime previous} args
    argnorm {{closedate 6} {closetime 6} {previous 1}} args
    argproc \$s $args {{closedate {}} {closetime {}} {previous N}}

    set now [clock seconds]
    if {$closedate == {}} { set closedate [clock format $now -format {%Y%m%d}] }
    if {$closetime == {}} { set closetime [clock format $now -format {%H%M%S}] }

    # build the input data
    set send {}
    set s lappend
    $s send "0 007"
    $s send "10 $cnf(account)"
    $s send "498 $cnf(meter)"
    $s send "3025 FDXG"
    $s send "1007 $closedate"
    $s send "1366 $closetime"
    $s send "1371 $previous"
    $s send "3014 1"
    $s send {99 {}}

    # send and receive
    set receive [send_receive [fedex::encode $send] ]
    set array ret
    fedex::decode $receive ret
    
    # -- parse decoded reply

    # none avail or errored out?
    if {[info exists ret(2)] == 1} {
        return $ret(3)
    }

    set retval {}
    loop var 0 [string trimleft $ret(1372) 0] 1 {
        set i [expr {$var+1}]
        set val $ret(1367-$i)
    }
    
    return $retval
}

#--- Track a package
# requires carrier and tracking number
# returns a list of signatories (signed for by, if exists) and scans (description, location, date)
# on error, returns the error message
proc fedex::trackPackage {carrier tracking} {
    variable cnf

    # build the input data
    set send {}
    set s lappend
    $s send "0 403"
    $s send "10 $cnf(account)"
    $s send "498 $cnf(meter)"
    $s send "3025 FDXG"
    $s send "1534 Y"
    $s send "1537 tracking"
    $s send "1538 0"
    $s send {99 {}}

    # send and receive
    set receive [send_receive [fedex::encode $send] ]
    set array ret
    fedex::decode $receive ret
    
    # -- parse decoded reply

    # none avail or errored out?
    if {[info exists ret(2)] == 1} {
        return $ret(3)
    }

    # "soft" error?
    if {[info exists ret(557)] == 1} {
        return $ret(559)
    }

    set retval {}
    loop var 0 [string trimleft $ret(1584) 0] 1 {
        set i [expr {$var+1}]
        if {[info exists ret(1706-$i)] == 1} {
            lappend retval [list $i "Signed for by $ret(1706-1)"]
        }
        if {[info exists ret(1715-$i)] == 1} {
            set val {}
            loop track 0 [string trimleft $ret(1715-$i) 0] 1 {
                set t [expr {$track+1}]
                if {[info exists ret(1159-$i-$t)] == 1} {
                    lappend val $ret(1159-$i-$t)
                }
                if {[info exists ret(1160-$i-$t)] == 1} {
                    lappend val "$ret(1160-$i-$t), $ret(1161-$i-$t), $ret(1164-$i-$t)"
                }
                if {[info exists ret(1162-$i-$t)] == 1} {
                    lappend val [clock format [clock scan "$ret(1162-$i-$t) $ret(1162-$i-$t)"] -format {%a %b %e %l:%M:%S %p %Y}]
                }
            }
            lappend retval [list $i $val]
        }
    }

    return $retval
}

if {[info commands locawyze] != {}} {locawyze fedex}
