package org.nakedobjects.runtime.transaction;

import org.nakedobjects.metamodel.commons.component.TransactionScopedComponent;
import org.nakedobjects.runtime.transaction.messagebroker.MessageBroker;
import org.nakedobjects.runtime.transaction.updatenotifier.UpdateNotifier;


/**
 * Used by the {@link NakedObjectTransactionManager} to captures a set of changes to be applied.
 * 
 * <p>
 * The protocol by which the {@link NakedObjectTransactionManager} interacts and uses the
 * {@link NakedObjectTransaction} is not API, because different approaches are used. For the server-side
 * <tt>ObjectStoreTransactionManager</tt>, each object is wrapped in a command generated by the underlying
 * <tt>ObjectStore</tt>. for the client-side <tt>ClientSideTransactionManager</tt>, the transaction simply
 * holds a set of events.
 * 
 * <p>
 * Note that methods such as <tt>flush()</tt>, <tt>commit()</tt> and <tt>abort()</tt> are
 * not part of the API.  The place to control transactions is through the {@link NakedObjectTransactionManager transaction manager},
 * because some implementations may support nesting and such like.  It is also the job of the
 * {@link NakedObjectTransactionManager} to ensure that the underlying persistence
 * mechanism (for example, the <tt>NakedObjectStore</tt>) is also committed.
 */
public interface NakedObjectTransaction extends TransactionScopedComponent {

    public static enum State {
        /**
         * Started, still in progress.
         * 
         * <p>
         * May {@link NakedObjectTransaction#flush() flush}, {@link NakedObjectTransaction#commit() commit} 
         * or {@link NakedObjectTransaction#abort() abort}.
         */
        IN_PROGRESS(true, true, true, false),
        /**
         * Started, but has hit an exception.
         * 
         * <p>
         * May not {@link NakedObjectTransaction#flush()} or {@link NakedObjectTransaction#commit() commit}
         * (will throw an {@link IllegalStateException}), but can only {@link NakedObjectTransaction#abort()
         * abort}.
         * 
         * <p>
         * Similar to <tt>setRollbackOnly</tt> in EJBs.
         */
        MUST_ABORT(false, false, true, false),
        /**
         * Completed, having successfully committed.
         * 
         * <p>
         * May not {@link NakedObjectTransaction#flush()} or {@link NakedObjectTransaction#abort() abort} or
         * {@link NakedObjectTransaction#commit() commit} (will throw {@link IllegalStateException}).
         */
        COMMITTED(false, false, false, true),
        /**
         * Completed, having aborted.
         * 
         * <p>
         * May not {@link NakedObjectTransaction#flush()}, {@link NakedObjectTransaction#commit() commit}
         * or {@link NakedObjectTransaction#abort() abort} (will throw {@link IllegalStateException}).
         */
        ABORTED(false, false, false, true);

        private final boolean canFlush;
        private final boolean canCommit;
        private final boolean canAbort;
        private final boolean isComplete;

        private State(final boolean canFlush, final boolean canCommit, final boolean canAbort, final boolean isComplete) {
            this.canFlush = canFlush;
            this.canCommit = canCommit;
            this.canAbort = canAbort;
            this.isComplete = isComplete;
        }

        /**
         * Whether it is valid to {@link NakedObjectTransaction#flush() flush} this {@link NakedObjectTransaction transaction}.
         */
        public boolean canFlush() {
            return canFlush;
        }

        /**
         * Whether it is valid to {@link NakedObjectTransaction#commit() commit} this {@link NakedObjectTransaction transaction}.
         */
        public boolean canCommit() {
            return canCommit;
        }

        /**
         * Whether it is valid to {@link NakedObjectTransaction#abort() abort} this
         * {@link NakedObjectTransaction transaction}.
         */
        public boolean canAbort() {
            return canAbort;
        }

        /**
         * Whether the {@link NakedObjectTransaction transaction} is complete (and
         * so a new one can be started).
         */
        public boolean isComplete() {
            return isComplete;
        }

    }

    /**
     * The state of this transaction.
     */
    State getState();

    /**
     * The underlying reason (if known) why the transaction was aborted.
     * 
     * <p>
     * Will be set if an exception is thrown while {@link #flush() flush}ing, 
     * {@link #commit() commit}ting or {@link #abort() abort}ing.
     */
    public RuntimeException getAbortCause();


    /**
     * The owning {@link NakedObjectTransactionManager transaction manager}.
     */
    NakedObjectTransactionManager getTransactionManager();

    /**
     * The {@link UpdateNotifier} for this transaction.
     */
    UpdateNotifier getUpdateNotifier();

    /**
     * The {@link MessageBroker} for this transaction.
     */
    MessageBroker getMessageBroker();


}
// Copyright (c) Naked Objects Group Ltd.
