#!/usr/bin/perl -w

# Copyright (c) 2001-2005 Extended Systems, Inc.
# Portions Copyright (c) 2005-2008, iAnywhere Solutions, Inc.
# All rights reserved. All unpublished rights reserved.
# Install script for the ODBC driver for Linux

use strict;         # force variable declarations
use DirHandle;

use File::Copy;     # for copying files

my $acefilename      = "libace.so.9.00.0.0";
my $aceverlinkname   = "libace.so.9.00";
my $acemainlinkname  = "libace.so";
my $alsfilename      = "libadsloc.so.9.00.0.0";
my $alsverlinkname   = "libadsloc.so.9.00";
my $alsmainlinkname  = "libadsloc.so";
my $odbcfilename     = "libadsodbc.so.9.00.0.0";
my $odbcverlinkname  = "libadsodbc.so.9.00";
my $odbcmainlinkname = "libadsodbc.so";

my $adscfgname = "adslocal.cfg";
my $productname = "Advantage ODBC Driver for Linux";
my $companyname = "iAnywhere Solutions, Inc.";
my $productversion;                  # Pull this from the file being installed

my $ownerinfo = "";

my $targetdir = "/usr/local/ads";    # default install location

# This is where the redistribute files go. It is also a flag on whether
# or not to install the files there. If empty string, no files are installed
# to the reditribute directory.
my $redistdir="";
my $ansilanguage = "Engl(Amer)";      # the chosen ANSI language
my $oemlanguage = "USA";              # OEM language name
my $locale = "<Default>";             # The specific locale if using machine
                                      # generated collation information.  If
                                      # <Default>, use the environment settings
                                      # as specified in setlocale(3)

my $creatednewdir = 0;                # indicates if we created a new directory
my $creatednewerrordir = 0;           # indicates if we created a new error directory
my $response;                         # temp variable
my $isrootuser = 1;                   # default to root user
my $HOME="";


# Change working directory to location of this script (since that is where
# the files we are dealing with should be)
my $newdir = $0;                  # get the command line that started the script
$newdir =~ s/(.*)\/.*$/$1/g;      # remove from last slash to end of string
if ( $newdir ne $0 )     # if it happens to just be an unadorned name, don't do anything
   {
   #
   chdir( $newdir ) || die( "Unable to change working directory to location of install script: $!" );
   }


# get the version number
$productversion = ExtractInternalVersion( "setup.files/redistribute/$acefilename" );

ClearScreen();

printf( "\nThis will install " . $productname . " (version " . $productversion . " )\n\n" );


if ( `whoami 2>/dev/null` !~ /^root$/ )
   {
   $isrootuser = 0;
   $HOME = `echo ~`;
   chomp( $HOME );

   # default target directory is user is not root
   $targetdir = "$HOME/ads";

   pp( "\nSince you are not logged in as root the Advantage ".
       "Client Engine will only be installed in the \"redistribute\" subdirectory." );
   } # User is not root

pp ( "\nHit Control-C at anytime to cancel this installation program" );
print "\nPress <Enter> to continue";
$_ = <STDIN>;

# License agreement
ClearScreen();
system( "cat license.txt | more " );
print "\nDo you agree with the license. [Y/n] ";
$_ = ReadResponse();
if ( $_ ne "" && $_ ne "Y" && $_ ne "y" )
   {
   exit;
   }

ClearScreen();
print "\n\nBe sure to close any Advantage enabled " .
      "application before proceeding.\n" .
      "Press <Enter> to continue";
<STDIN>;

# Get the answers
PromptForInstallInfo();

# Set the redistribute directory
$redistdir="$targetdir/odbc/redistribute";

print "\nTrying to create: $targetdir/odbc\n";
# Create the directory as whatever user we are logged in as.
$creatednewdir = CreateDirectory( "$targetdir/odbc" );
print "\nCreated the directory\n";

# see if there is enough space
CheckDiskSpace( $targetdir );

# Make copies of the existing files
BackupFiles();

# Copy the daemon to the target location
CopyFilesToDestination();


SetLinks();

# All done

pp( "\n\nThe $productname installation is complete. " );

pp( "Help files and other documentation can be found in the $targetdir subdirectories." );

sub ReadResponse
{
   $_ = <STDIN>;
   chomp( $_ );
   $_ =~ s/^\s*//; # Remove proceeding white space

   return $_
}



sub DoCFGMerge
{
   my $origmtime;
   my $origatime;
   my %newadsconf;
   my %oldadsconf;
   my %newadsinifile;

   my ( $oldadscfg, $newadscfg ) = @_;

   if ( !( -f $newadscfg ) )
      {
      # This should never be the case since we just copied it here.
      return;
      }

   # extract time/date info of ads conf file
   ($origatime, $origmtime) = ( stat( $newadscfg ))[8, 9];

   # At this point, we have the new adslocal.cfg file and possibly the original adslocal.cfg
   # in the target directory (renamed as adslocal.cfg.old). We need to
   # merge all this stuff together into a single glorious adslocal.cfg.
   # The method for doing the merging is pretty much brute force - slurp up the
   # contents of the files into hash variables and then print out the new file.
   if ( -e "$oldadscfg" )
      {
      # There is an existing file
      %oldadsconf = ReadINISection( $oldadscfg, "" );
      %newadsconf = ReadINISection( $newadscfg, "" );

      # combine the two (pass the hashes by reference)
      MergeConfFiles( \%oldadsconf, \%newadsconf );
      }
   else
      {
      # there is just the new conf file
      %newadsconf = ReadINISection( $newadscfg, "" );
      }

   # For ALS the <Default> setting needs to be an empty string
   if ( "$ansilanguage" eq "<Default>")
      {
      $ansilanguage = "";
      }

   # Put in the languages    $newconfref->{ "$key" } =
   $newadsconf{lc("ANSI_CHAR_SET")} = "$ansilanguage";
   $newadsconf{lc("OEM_CHAR_SET")} = "$oemlanguage";

   # Copy the cfg file to the redistribute directory.
   WriteConfFile( $newadscfg , %newadsconf );

   # set the modification times on the file to the install version
   utime( $origatime, $origmtime, $newadscfg );

} # DoCFGMerge





# Clear the screen
sub ClearScreen
{
   # this evaluates the "clear" command
   system( "clear" );
}



# Extract the version information from the given file name using the ident
# string stored in Advantage binaries.
sub ExtractInternalVersion
{
   my ($filename) = @_;

   my $ver;

   # search for it in the file
   $ver = FileHasString( $filename, "EsIAx!@#" );

   # bail if we didn't find it
   if ( !$ver )
      {
      printf( "\nWarning - The internal version number was not found in $filename\n\n" );
      return "";
      }

   # we now have a long string of junk that contains the string in the middle
   # of it.  Extract it from the string.  The part we want is the ([\w\.]+)
   # portion which finds all text of alphanumeric characters and "."s following
   # the ident string.
   $ver =~ s/(.*EsIAx!@# )([\w\. ]+)(.*)/$2/;
   chomp( $ver );

   return $ver;
}  # ExtractInternalVersion



# This prompts for user input.  It just fills in the global variables
sub PromptForInstallInfo
{
   my $bDone = 0;
   my $bvalidproduct = 0;
   my $pwd;
   my $DefaultPath;

   # Save the current path as the default path.
   $DefaultPath = $targetdir;

   while ( !$bDone )
      {
      $bvalidproduct = 0;

      # Reset the path in case it was changed.
      $targetdir = $DefaultPath;

      print "\n\n\nEnter the destination path: ($targetdir) ";
      $_ = ReadResponse();
      $targetdir = $_ if ( $_ ne "" );

      if ( $targetdir =~ /.*\/$/ )
         {
         # remove from last slash to end of string
         $targetdir =~ s/(.*)\/.*$/$1/g;
         }

      if ( $targetdir =~ /^~/ )
         {
         # it starts with a ~ so expland it.
         $targetdir = tildeexpand( $targetdir );
         }

      # Resolve any relative paths.
      chop($pwd = `pwd`);
      ($targetdir =~ m|^\/|) || (substr($targetdir, 0, 0) = $pwd."/");


      # Get the language (character set) information. This is needed for all
      # users since they will all have a adslocal.cfg in the redistribute
      # directory to update.
      PromptForLanguages();

      # Show them their choices and let them change it if desired
      ClearScreen();
      print "\nThese are the current install selections:\n";

      print "   Destination path: $targetdir\n";

      print "   ANSI Language: $ansilanguage\n"; # if ( $ansilanguage !~ /default/i );
#         print "   Locale for default character set generation: $locale\n" if  ( $ansilanguage =~ /default/i );
      print "   OEM Language: $oemlanguage\n";


      print "\nInstall with these choices? [Yn]: ";
      $_ = ReadResponse();
      $bDone = 1 if (( $_ =~ /^y/i ) || ( $_ eq "" ))

      }  # while not done

}  # PromptForInstallInfo


# stupid function to write a "paragraph" to stdout with word wrapping at
# 72 characters.  There are probably clever/cool/weird ways to do this in
# perl, but I don't know them.  I read in a couple NG posts that you can
# just use formats which will do the word wrapping.  Any format I tried,
# though, just truncated the text or broke at the end of the terminal screen.
sub pp     # as in "print paragraph"  (it's lame but short)
{
   my ( $longtext ) = @_;

   # put a space on the end - otherwise it sometimes wraps the last word
   $longtext = $longtext . " ";

   # Wrap at 72 chars - This is from a David Bell '96 news group post.
   $longtext =~ s/(.{1,72})\s+/$1\n/g;

   print $longtext;
}




# filename is what the link will be pointing to. It can be another link instead
# of a file name.
# linkname is the new link.
# If bupdate is 0 it is removed if it exist and the new link created. If it is 1
# it is only updated if the filename is the same version or newer.
sub SetSpecificLink
{
   my ( $filename, $linkname, $bupdate ) = @_;
   my $oldverlinkname = "";


   if ( $bupdate == 0 )
      {
      # Create the version symbolic link. This link is always
      # created regardless if it is an older version being installed.
      unlink ( $linkname );
      symlink( $filename, $linkname ) ||
         die "Cannot create symbolic link to $filename";
      }
   else
      {
      if ( !( -f $linkname ))
         {
         # Link is not there so create one.
         symlink( $filename, $linkname ) ||
                        die "Cannot create symbolic link to $filename";
         }
      else
         {
         # Link exist so get the file name it is pointing to.
         $oldverlinkname = readlink $linkname;
         if ( !( "$oldverlinkname" eq "" ) )
            {
            # Got a file name. Since the version is in the file name, see
            # if the link needs updating. If the file names are the same
            # then leave the link alone since it is already correct. If
            # the file name is a newer version, leave it alone as well
            # since this link always points to the highest version of the
            # file.
            if ( "$oldverlinkname" lt $linkname )
               {
               # Update the link
               unlink( $linkname );
               symlink( $filename, $linkname ) ||
                        die "Cannot create symbolic link to $filename";
               }
            } # Got a file name that the link is pointing to
         } # File doesn't exists

      } # if $bupdate
} # SetSpecificLink




# SetLinks
# This needs to be called after the install is completed. This function
# updates/creates the major links like the "libace.so" link if the link
# doesn't exist or if the revision is the same or newer than the
# pre-existing file.
# The revision link, "for example libace.so.6.11", is always
# updated/created regardless pre-existing versions.
# Also, since this only runs if the install was successful, the
# files should exist but check to make sure.
#
sub SetLinks
{
   my $i;
   my $verlinkname;
   my $mainlinkname;
   my $oldverlinkname="";



   if ( $isrootuser )
      {
      SetSpecificLink( "/usr/lib/$acefilename", "/usr/lib/$aceverlinkname", 0 );
      SetSpecificLink( "/usr/lib/$alsfilename", "/usr/lib/$alsverlinkname", 0 );
      SetSpecificLink( "/usr/lib/$aceverlinkname", "/usr/lib/$acemainlinkname", 0 );
      SetSpecificLink( "/usr/lib/$alsverlinkname", "/usr/lib/$alsmainlinkname", 0 );
      SetSpecificLink( "/usr/lib/$odbcfilename", "/usr/lib/$odbcverlinkname", 0 );
      SetSpecificLink( "/usr/lib/$odbcverlinkname", "/usr/lib/$odbcmainlinkname", 0 );
      } # root user

   # Setup the same ACE and ALS links in the redistribute directory
   SetSpecificLink( "$redistdir/$acefilename", "$redistdir/$aceverlinkname", 0 );
   SetSpecificLink( "$redistdir/$alsfilename", "$redistdir/$alsverlinkname", 0 );
   SetSpecificLink( "$redistdir/$aceverlinkname", "$redistdir/$acemainlinkname", 0 );
   SetSpecificLink( "$redistdir/$alsverlinkname", "$redistdir/$alsmainlinkname", 0 );
   SetSpecificLink( "$redistdir/$odbcfilename", "$redistdir/$odbcverlinkname", 0 );
   SetSpecificLink( "$redistdir/$odbcverlinkname", "$redistdir/$odbcmainlinkname", 0 );


} # SetLinks


# Read the available character set names from ansi.chr and return them in an array
sub ReadANSINames
{
   my @charsets;   # return value
   my $name;       # current character set name
   my $inuse;      # inuse flag for the character set
   my $offset;     # offset of character set in file
   # This defines the "record" of a header value
   my $recformat = 'C A11 L';   # Unsigned character, 11 byte name, 4 byte offset
   my $recwidth  = 16;          # width of a record
   my $i;
   my $rec;        # The raw record
   my $numrecs = 25; # The current ansi.chr has up to 25 character sets
   my $totalsets = 0;

   open( CHARSET, "setup.files/redistribute/ansi.chr" ) || die "Unable to open ansi.chr:  $!";

   for ( $i = 0; $i < $numrecs; $i++ )
      {
      # read the raw record from the table
      read( CHARSET, $rec, $recwidth ) == $recwidth || die "Unable to read record $i from ansi.chr: $!";

      # unpack the record into usable variables
      ($inuse, $name, $offset) = unpack( $recformat, $rec );

      # if it is in use, add it to the array
      $charsets[$totalsets++] = $name if $inuse;
      }

   close( CHARSET );

   return @charsets;

}  # ReadANSINames


# Read the available character set names from extend.chr and return them in an array
# Pretty much identical to the above routine to read the ansi names.  It could
# be a single function with some parameters, but I'm lazy.
sub ReadOEMNames
{
   my @charsets;   # return value
   my $name;       # current character set name
   my $inuse;      # inuse flag for the character set
   my $offset;     # offset of character set in file
   # This defines the "record" of a header value
   my $recformat = 'C A9 L';   # Unsigned character, 9 byte name, 4 byte offset
   my $recwidth  = 14;         # width of a record
   my $i;
   my $rec;        # The raw record
   my $numrecs = 50; # The current ansi.chr has up to 25 character sets
   my $totalsets = 0;

   open( CHARSET, "setup.files/redistribute/extend.chr" ) || die "Unable to open extend.chr:  $!";

   for ( $i = 0; $i < $numrecs; $i++ )
      {
      # read the raw record from the table
      read( CHARSET, $rec, $recwidth ) == $recwidth || die "Unable to read record $i from ansi.chr: $!";

      # unpack the record into usable variables
      ($inuse, $name, $offset) = unpack( $recformat, $rec );

      # if it is in use, add it to the array
      $charsets[$totalsets++] = $name if $inuse;
      }

   close( CHARSET );

   return @charsets;

}  # ReadOEMNames



# Given a list of names and a response, check to see if the response is in
# the list (or is a valid numeric 1 based reference to an item in the list).
# The obvious way (to me) of doing this is to pass the response by reference
# and change the value.  I wasn't smart enough to make that work, so I return
# two values from the function, which in itself is a pretty cool concept.
sub CheckListResponse
{
   my ( $response, @choices ) = @_;

   my $validanswer = 0;
   my $index;


   printf "!!! Test:  %s, response = %s\n", $choices[6], $response ;

   if ( $response =~ /^[\d]+$/ )
      {
      # They chose a numeric value
      if ( $response > 0 && $response <= @choices )
         {
         $response = $choices[$response - 1];
         $validanswer = 1;
         }
      }  # if numeric input
   else
      {
      # apparently typed in a name - make sure it is in the list
      for ( $index = 0; $index < @choices; $index++ )
         {
         if ( $response =~ /$choices[$index]/i )
            {
            $response = $choices[$index];
            $validanswer = 1;
            last;
            }
         }
      }

   # return the two values in a list
   return ( $validanswer, $response );

}  # CheckListResponse




# Prompt the user for which language from the ansi.chr and extend.chr files
# they want.  This is for international installs.
sub PromptForLanguages
{
   my @ansicharsetnames;     # array of character set names from ansi.chr
   my @oemcharsetnames;      # array of character set names from extend.chr
   my $index;
   my $validanswer;          # true/false
   my $response;             # the response from the user



   # Make sure the ansi.chr exists
   if ( -e "setup.files/redistribute/ansi.chr" )
      {
      @ansicharsetnames = ReadANSINames();
      }
   else
      {
      # This shouldn't happen - if it does, it means that they will have to
      # choose <Default> as the ansi language and we will generate a collation
      # language for them.
      pp "\nWarning:  The file ansi.chr does not exist.  The installation should " .
         "have included this file.  You will have to allow the installation " .
         "program to generate the ANSI character set.";
      printf( "\nPress <enter> to continue " );
      <STDIN>;
      }


   # make sure extend.chr exists
   if ( -e "setup.files/redistribute/extend.chr" )
      {
      @oemcharsetnames = ReadOEMNames();
      }
   else
      {
      # This shouldn't happen - The script uses the existence of extend.chr as
      # part of the decision that this is an international install, which is
      # the only way we should get to this routine.  So something is goofed up.
      pp "Warning.  The extend.chr file does not exist.  The installation should " .
         " have included this file.  It is required to complete the installation.\n";
      exit 1;
      }


   # Need to prompt for the information
   $validanswer = 0;
   ClearScreen();

   while ( !$validanswer )
      {

      # Get the ANSI language
      pp( "\nPlease select an ANSI character set to use with the Advantage Local Server." );
      pp( "\nIf <Default> is chosen, the installation program will generate ".
#          "a character set to use based on a locale that you can choose." );
          "a character set to use based on the current locale." );
      pp( "\nSelecting a specific ANSI language for all Advantage installs (including the " .
          "Advantage Database Server) will guarantee the ANSI character sets used by all " .
          "Advantage applications will be the same." );
      pp( "\nThis setting does not apply to tables opened with OEM as the ".
          "specified character set type or to Advantage DOS applications." );

      printf( "%5d - %s\n", 0, "<Default>" );

      for ( $index = 0; $index+1 < @ansicharsetnames; $index+=2 )
         {
         printf( "%5d - %-11s\t", $index + 1, $ansicharsetnames[$index] );
         printf( "%5d - %s\n", $index + 2, $ansicharsetnames[$index +1] );
         }

      if ( $index < @ansicharsetnames )
         {
         printf( "%5d - %s\n", $index + 1, $ansicharsetnames[$index] );
         }

      print "\nPlease select a language: ($ansilanguage) ";
      $response = ReadResponse();;

      if ( $response eq "" )
         {
         # No change - leave ansi language the same
         $validanswer = 1;
         }
      else
         {
         # Now make sure they entered a valid value
         if ( $response =~ /^0$|default/i )
            {
            # They chose default
            $ansilanguage = "<Default>";
            $validanswer = 1;
            }
         else
            {
            # Something other than default - extract the result - pass the
            # list and the response to this function.  It returns the boolean
            # indicating if it is valid and the converted response value if
            # it is valid.
            ( $validanswer, $response ) = CheckListResponse( $response, @ansicharsetnames );

            $ansilanguage = $response if $validanswer;

            }  # else not default
         }  # else response entered

      if ( !$validanswer )
         {
         ClearScreen();
         pp( "\nYour choice was not valid.  Please enter a numeric value from " .
             "the list or type the desired name." );
         }

      }  # while not valid ansicharset


      # If they chose <default>, need to request a locale.
      if ( $ansilanguage =~ /default/i )
         {
#         ClearScreen();
#         pp( "\nSince you chose $ansilanguage for the ANSI language, you can provide " .
#             "a locale setting that will be used to generate the character set. " .
#             "The valid locale values differ by workstation and depend on the " .
#             "specific Linux installation.  The actual locale definitions are " .
#             "usually stored in /usr/share/i18n/locales. " .
#             "If you specify <Default>, the current environment settings ".
#             "(e.g., LC_ALL, LC_COLLATE, and LANG) are used. ".
#             "See the locale(5) and setlocale(3) manpages for more information. " );

#         print "\nPlease enter the locale: ($locale) ";

#         $_ = ReadResponse();;
         $_ = "";
         $locale = $_ if $_ ne "";

         # convert to <Default> for consistency if needed
         $locale = "<Default>" if $locale =~ /default/i;
         }



   # prompt for OEM language
   $validanswer = 0;
   ClearScreen();

   while ( !$validanswer )
      {
      pp( "\nThe following only applies if your Advantage applications use OEM/Localized " .
          "character sets." );
      pp( "\nPlease select an OEM character set to use with the Advantage Local Server. " );
      pp( "\nSelecting a specific OEM/Localized character set for all Advantage " .
          "installs (including the Advantage Database Server) will guarantee the " .
          "OEM/Localized character sets used by all Advantage applications will be the same." );

      for ( $index = 0; $index+1 < @oemcharsetnames; $index+=2 )
         {
         printf( "%5d - %-11s\t", $index + 1, $oemcharsetnames[$index] );
         printf( "%5d - %s\n", $index + 2, $oemcharsetnames[$index +1] );
         }

      if ( $index < @oemcharsetnames )
         {
         printf( "%5d - %s\n", $index + 1, $oemcharsetnames[$index] );
         }

      print "\nPlease select a language: ($oemlanguage) ";
      $response = ReadResponse();;

      # if no response, just use the existing contents of the oemlang variable
      $response = $oemlanguage if $response eq "";

      # make sure the response is in the list of available choices
      ( $validanswer, $response ) = CheckListResponse( $response, @oemcharsetnames );

      # assign the response if it was valid
      $oemlanguage = $response if $validanswer;

      if ( !$validanswer )
         {
         ClearScreen();
         pp( "\nYour choice was not valid.  Please enter a numeric value from " .
             "the list or type the desired name." );
         }

      }  # while not valid oem response

}  # PromptForLanguages



# copy a file
sub CopyFile
{
   # Get the parameters:  source file, destination file, desired permissions
   my ( $source, $dest, $permissions ) = @_;

   my $origmtime;
   my $origatime;

   # extract time/date info to restore after the copy (want 8th and 9th params of stat)
   ($origatime, $origmtime) = ( stat( $source ))[8, 9];

   # copy the daemon
   copy( $source, $dest ) ||
                die  "Failed to copy $source to $dest.  Please verify that you have the correct permissions:  $!";

   # Make it executable by the owner
   chmod( $permissions, $dest );

   # restore the time
   utime( $origatime, $origmtime, "$dest" );
}




# Copy a directory to given location
# note: This will not get subdirectories
sub Copydirectory
{

   my ( $curdir, $dest ) = @_;

   my @files;
   my $file;

   my $handle = new DirHandle or die "Couldn't create directory Handle: $!";

   opendir $handle, $curdir or die "Couldn't open directory $curdir: $!";

   @files = readdir $handle or die "Couldn't read directory $curdir: $!";

   closedir $handle or die "Couldn't close directory $curdir: $!";

   # Create the target directory
   CreateDirectory( "$dest" );

   # Now copy each file in this directory
   foreach $file ( @files )
      {
      if ( !( -d $file ))
         {
         # copy the single file
         CopyFile( "$curdir/$file", "$dest/$file", 0644 );
         }
      }  # for each file

}  # Copydirectory


# Copy the files to the destination location
sub CopyFilesToDestination
{

   printf( "Copying files...\n" );

   CopyFile( "license.txt", "$targetdir/odbc/license.txt", 0644 );

   # install the help files
   system( "tar --no-same-owner -zxf setup.files/help.tar.gz -C$targetdir" );

   # Set up the help file index html file
   CreateMainHelp( $targetdir );


   if ( $isrootuser )
      {
      # copy the binary
      CopyFile( "setup.files/redistribute/$acefilename", "/usr/lib/$acefilename", 0744 );
      CopyFile( "setup.files/redistribute/$alsfilename", "/usr/lib/$alsfilename", 0744 );
      CopyFile( "setup.files/redistribute/$odbcfilename", "/usr/lib/$odbcfilename", 0644 );
      CopyFile( "setup.files/redistribute/ansi.chr", "/etc/ansi.chr", 0644 );
      CopyFile( "setup.files/redistribute/extend.chr", "/etc/extend.chr", 0644 );
      CopyFile( "setup.files/redistribute/$adscfgname", "/etc/$adscfgname", 0644 );
      CopyFile( "setup.files/redistribute/adscollate.adt", "/etc/adscollate.adt", 0644 );
      CopyFile( "setup.files/redistribute/adscollate.adm", "/etc/adscollate.adm", 0644 );

      DoCFGMerge( "/etc/$adscfgname.old", "/etc/$adscfgname" );
      } # root user

   # put a copy of the files in the redistribute directory.
   $creatednewdir = CreateDirectory( "$redistdir" );
   CopyFile( "setup.files/redistribute/$acefilename", "$redistdir/$acefilename", 0744 );
   CopyFile( "setup.files/redistribute/$alsfilename", "$redistdir/$alsfilename", 0744 );
   CopyFile( "setup.files/redistribute/ansi.chr", "$redistdir/ansi.chr", 0644 );
   CopyFile( "setup.files/redistribute/extend.chr", "$redistdir/extend.chr", 0644 );
   CopyFile( "setup.files/redistribute/$adscfgname", "$redistdir/$adscfgname", 0644 );
   CopyFile( "setup.files/redistribute/$odbcfilename", "$redistdir/$odbcfilename", 0644 );
   CopyFile( "setup.files/redistribute/adscollate.adt", "$redistdir/adscollate.adt", 0644 );
   CopyFile( "setup.files/redistribute/adscollate.adm", "$redistdir/adscollate.adm", 0644 );

   # Do the conf file copy/merge
   DoCFGMerge( "$redistdir/$adscfgname.old", "$redistdir/$adscfgname" );

   ClearScreen();

}  # CopyFilesToDestination



# This creates the index.htm file that has the links to the
# other help files. We can't simply copy a new file there in
# case another install (for example ARC) has updated it.
sub CreateMainHelp
{
   my ( $filename ) = @_;
   my $newindexfile;
   my $line;
   my $berrordone = 0;
   my $badvantagedone = 0;
   my $bodbcdone = 0;
   my $bacedone = 0;
   my $bfoundlinks = 0;


   $filename = "$filename/help/index.htm";

   if ( !( -e $filename ))
      {
      # File doesn't exist so just create one.
      $newindexfile = "<HTML>\n" .
                      "<BODY BGCOLOR=white>\n\n" .
                      "<CENTER>\n" .
                      "<H2>\n\n" .
                      "<P><A HREF=\"./Advantage/advantage.htm\">Advantage General Help</A>\n" .
                      "<P><A HREF=\"./ADSERROR/adserror.htm\">Advantage Error Codes</A>\n\n" .
                      "<P><A HREF=\"./odbc/adsodbc.htm\">Advantage ODBC Guide</A>\n\n" .
                      "<P><A HREF=\"./ace/ace.htm\">Advantage Client Engine</A>\n\n" .
                      "</CENTER>\n\n" .
                      "</BODY>\n" .
                      "</HTML>\n";

      WriteFile( $filename, $newindexfile );
      }
   else
      {
      # attempt to open the file
      open( INIFILE, $filename ) || return 0;

      # search for the section
      while ( $line = <INIFILE> )
         {
         # if there is a link find out if it links to one of the help files
         # we are installing.
         if ( $line =~ /href/i )
            {
            if ( $line =~ /advantage\/advantage.htm/i )
               {
               # The main advantage help file is in there
               $badvantagedone = 1;
               }

            if ( $line =~ /adserror\/adserror.htm/i )
               {
               # The error help file is in there
               $berrordone = 1;
               }
            if ( $line =~ /odbc\/adsodbc.htm/i )
               {
               # The main advantage help file is in there
               $bodbcdone = 1;
               }

            if ( $line =~ /ace\/ace.htm/i )
               {
               # The error help file is in there
               $bacedone = 1;
               }
            $bfoundlinks = 1;

            }
         else
            {
            if ( $bfoundlinks && $line ne "" )
               {
               # We are past the links and it isn't just a blank
               # line between the links. Now set the links that
               # weren't found in the file.
               if ( $badvantagedone == 0 )
                  {
                  $newindexfile = "$newindexfile<P><A HREF=\"" .
                     "./Advantage/advantage.htm\">Advantage General Help</A>\n";
                  }

               if ( $berrordone == 0 )
                  {
                  $newindexfile = "$newindexfile<P><A HREF=\"" .
                     "./ADSERROR/adserror.htm\">Advantage Error Help</A>\n";
                  }

               if ( $bodbcdone == 0 )
                  {
                  $newindexfile = "$newindexfile<P><A HREF=\"" .
                     "./odbc/adsodbc.htm\">Advantage ODBC Guide</A>\n\n";
                  }

               if ( $bacedone == 0 )
                  {
                  $newindexfile = "$newindexfile<P><A HREF=\"" .
                     "./ace/ace.htm\">Advantage Client Engine</A>\n\n";
                  }


               # We are done adding the links
               $bfoundlinks = 0;


               } # Past the link section
            } # if link section

         $newindexfile = "$newindexfile$line";
         } # # Read all the lines in the file

      WriteFile( $filename, $newindexfile );
      } # index file doesn't exist

}# CreateMainHelp





# write out the given ads conf file
sub WriteConfFile
{
   my ( $filename, %newconf ) = @_;

   # try to open the file for output.  Not sure if this should be a fatal
   # error.  If we got this far (we were able to copy files into the directory
   # where we are trying to write to), then I'm not sure what would cause
   # the failure, but it seems unlikely that anything good is happening, so
   # just die for now.
   open( CONFFILE, "> $filename" ) || die "Unable to open $filename:  $!";


   # Loop through the entries in the order we read them
   my $current = 1;
   my $key;

   while ( 1 )
      {
      # bail out if there are no more entries
      last if !$newconf{ "$current" };

      $key = $newconf{ "$current" };

      # print the comment to the file
      printf( CONFFILE "%s", $newconf{ "$key.comment" } );

      # print the key/value (key in upper case)
      printf( CONFFILE "%s=%s\n", uc( $key ), $newconf{ "$key" } );

      $current++;
      }  # while


   close( CONFFILE );

}  # WriteConfFile



# write out the given string to a file
sub WriteFile
{
   my ( $filename, $newconf ) = @_;

   # try to open the file for output.  Not sure if this should be a fatal
   # error.  If we got this far (we were able to copy files into the directory
   # where we are trying to write to), then I'm not sure what would cause
   # the failure, but it seems unlikely that anything good is happening, so
   # just die for now.
   open( CONFFILE, "> $filename" ) || die "Unable to open $filename:  $!";

   printf( CONFFILE "%s", $newconf );

   close( CONFFILE );

}  # WriteFile




# Merge two ads.conf files that are stored in the hash produced by the
# ReadIniSection code.  All this does is takes every key/value pair for
# actual entries from the old hash and stores them in the new hash (presumably
# overwriting the existing value).  The hash actually has three entries for
# every conf file entry.  It has the key/value.  But it also has a key.comment/comment
# entry and an n/key entry for ordering.
#
# If there is a value in the old file that is not in the new file, don't copy it
# because it is apparently obsolete.
sub MergeConfFiles
{
   # the hashes are expected to be passed by reference.  The second one
   # is the new file (it is modified in place)
   my ($oldconfref, $newconfref) = @_;


   # A couple ways come to mind to do this.  One is to simply do a "foreach" for
   # all entries in the hash and ignore the "key.comment" and "n" keys.  The
   # other way (the one I'm using) is to loop 1 to n until we run past the end
   # of the "n" keys.

   my $current = 1;
   my $key;
   my $value;

   while ( 1 )
      {
      # bail out if there are no more entries
      last if !$oldconfref->{ "$current" };

      $key = $oldconfref->{ "$current" };
      $value = $oldconfref->{ "$key" };

      # store it in the new conf (if it already exists)
      $newconfref->{ "$key" } = $value if defined( $newconfref->{ "$key" } );

      $current++;
      }  # while

} # MergeConfFiles



# Backup any existing files in the target directory that we plan to overwrite
# This uses the global variables directly
sub BackupFiles
{
   my $cfgname = "$redistdir/$adscfgname";

   # Backup the file in the redistribute dir.
   if ( -e "$cfgname" )
      {
      # And rename the config file
      rename( "$cfgname", "$cfgname.old" )
                 || die "Unable to rename $cfgname to $cfgname.old:  $!";
      }

   # If root, backup the file in the etc dir.
   if ( $isrootuser )
      {
      $cfgname = "/etc/$adscfgname";
      } # root user

   if ( -e "$cfgname" )
      {
      # And rename the config file
      rename( "$cfgname", "$cfgname.old" )
                 || die "Unable to rename $cfgname to $cfgname.old:  $!";
      }
}  # BackupFiles



# Create the given directory if it does not exist.  Return 1/0 indicating if
# we did create the new directory.
sub CreateDirectory
{
   my ($location) = @_;
   my $cnt=0;

   if ( ! -d $location )
      {
      # use the system mkdir command because it will create parent directories
      # if needed.
      my $sysret = system( "mkdir -p $location" );
      if ( $sysret != 0 )
         {
         $sysret /= 256;  # documentation says it must be divided by 256 for true code
         die "Unable to create the directory $location.  Make sure you have the correct permissions.  Error code = $sysret";
         }

      # set the permissions so that users can see and traverse the directory
      $cnt = chmod( 0755, $location );

      return 1;
      }

   # did not create a new directory
   return 0;

}  # CreateDirectory




# Check for the amount of disk space
sub CheckDiskSpace
{
   my ($location) = @_;

   unless ( open ( DF, "df -k $location|" ) )
      {
      printf ( "Unable to get disk space information for $location.  Error is $!\n" );
      return;
      }

   while ( <DF> )
      {
      if ( /No such file/ )
         {
         printf( "Unable to get disk space information for $location.\n" );
         return;
         }

      # TBD - do we really need this information?  If so, we can parse the
      # output of this command and look for the amount of space left and
      # compare it to the various file sizes.  (adsd and the backup files
      # we may make).  The alternative is that the install will fail somewhere
      # along the line with a fairly obvious error any way.
      }

   close( DF );
}  # CheckDiskSpace




# Search a file for the existence of a string.  Return the string if found.
sub FileHasString
{
   my ( $filename, $search ) = @_;

   my $hasstring = 0;


   open( SEARCHFILE, $filename ) || die "Unable to open $filename:  $!";

   while ( <SEARCHFILE> )
      {
      if ( /$search/ )
         {
         $hasstring = $_;
         last;
         }
      }

   close( SEARCHFILE );

   return $hasstring;
}  # FileHasString






# Slurp in a section of an INI style file.  It returns a hash
# of the key/value pairs.
# Input params are the filename and the section to read
# The section (second parameter) can be empty in which case it just starts
# reading pairs at the top and quit if it runs into a section.
#
# NOTE - all key names are converted to lower case.
#
# As it reads the file it puts all comment/whitespace stuff into the hash
# as well as the entries.  The assumption is that the comments precede the
# entry.  If the entry is something like "thing=stuff", then the hash will
# contain the key "thing" with value "stuff" and a key named "thing.comments"
# with all the lines of comments preceding the entry.
#
# This also kludges in a method for extracting the values from the hash in
# the order we read them.  It adds a hash entry of the form "number"/"key".  So
# if the above example "thing=stuff" was the first entry.  There will be a
# "1"/"thing" entry in the hash.
#
# All of this probably constitutes severe hash-abuse, but the files we are
# dealing with are reasonably small, so it should be pretty fast.
sub ReadINISection
{
   # assign params
   my ( $filename, $section ) = @_;

   my %pairs;     # hash of key/value pairs  (the return value)
   my $line;      # current line from the file
   my @keyval;    # current key/value pair
   my $comments = "";
   my $currententry = 1;

   # make sure there are two arguments (sanity check on the calls)
   ( @_ == 2 ) || die "ReadINISection requires two parameters";

   # attempt to open the file
   open( INIFILE, $filename ) || return 0;

   # search for the section
   while ( $line = <INIFILE> )
      {
      if (( lc( $line ) =~ /\[$section\]/i ) || ( $section eq "" ))
         {
         # we want the first line if not doing a specific section
         $comments = $line if $section eq "";

         # Found the section we are looking for - now loop until end of section
         while ( $line = <INIFILE> )
            {
            if (( $line =~ /^\s*$/ ) || ( $line =~ /^\s*[;#]/ ))
               {
               # it is a comment character  (starts with # or ; after optional white space)
               $comments = $comments . $line;
               next;
               }

            # remove carriage return/line feed chars.  Chomp will not remove
            # the stupid DOS ^M.  This following substitution gets both.
            $line =~ s/[\015\012]+$//;

            if ( $line =~ /^\s*\[.*\]/ )  # note:  \s matches white space
               {
               # found a new section, bail out
               last;
               }

            @keyval = split( "=", $line, 2 );
            if ( @keyval == 2 )
               {
               # remove leading and trailing blank spaces
               $keyval[0] =~ s/^\s+//;
               $keyval[0] =~ s/\s+$//;
               $keyval[1] =~ s/^\s+//;
               $keyval[1] =~ s/\s+$//;

               # make the key lower case so we consistently read it
               $keyval[0] = lc( $keyval[0] );

               # store the key/value pair
               $pairs{ $keyval[0] } = $keyval[1] ;

               # store the key.comment/comment pair
               $pairs{ "$keyval[0].comment" } = $comments;

               # store the ordering info hash element
               $pairs{ "$currententry" } = $keyval[0];

               $comments = "";    # reset the comment collector variable

               # next entry
               $currententry++;

               }
            }  # while in the section
         }  # if in the section
      }  # while not eof



   close( INIFILE );

# debug - print what we found
#    my $sKey;
#    foreach $sKey ( keys %pairs )
#       {
#       printf( "'%s'='%s'\n", $sKey, $pairs{ $sKey } );
#       }
#   <STDIN>;

   # return the hash of keys and values
   return %pairs;

}  # ReadINISection



# This function adds the section and key to the file if it
# isn't there. Otherwise it is just updated.
sub AddKeyToSection
{
   # assign params
   my ( $filename, $section, $key, $value ) = @_;

   my $line;      # current line from the file
   my $filedata = "";
   my $foundsection = 0;
   my $foundkey = 0;


   # make sure there are two arguments (sanity check on the calls)
   ( @_ == 4 ) || die "AddKeyToSection requires four parameters";

   # attempt to open the file for appending
   open( INIFILE, $filename ) || return 0;

   # search for the section
   $line = <INIFILE>;

   while ( $line )
      {
      # if the sections name is not the same, read it in.
      if ( !( lc( $line ) =~ /^\[$section\]/i ))
         {
         # This is not the section we are looking for
         $filedata = "$filedata$line";
         $foundsection = 0;
         }
      else
         {
         # This is the section we are looking for
         $filedata = "$filedata$line";
         $foundsection = 1;
         }

      # loop until end of section
      while ( $line = <INIFILE> )
         {
         if ( $line =~ /^\s*\[.*\]/ )  # note:  \s matches white space
            {
            # found a new section, bail out
            last;
            }
         else
            {

            if ( "$line" ne "" && $foundsection && ( lc( "$line" ) =~  /^$key/i ) )
               {
               # Found the key to update
               $filedata = "$filedata$key=$value\n";
               $foundkey = 1;
               }
            else
               {
               $filedata = "$filedata$line";
               }
            } # Section has brackets

         }  # while in the section

      if ( $foundsection && !$foundkey )
         {
         # Found the section but not the key, add it.
         $filedata = "$filedata$key=$value\n";
         $foundsection = 0;
         $foundkey = 1;
         }

      }  # while not eof

   if ( !$foundkey )
      {
      # The section was not found add it and the key.
      $filedata = "$filedata\[$section]\n$key=$value\n";
      }

   close( INIFILE );

   # Write the new files out.
   WriteFile( $filename, $filedata );

}  # AddKeyToSection




# This function reads in all the sections but the one specified.
sub SkipINISection
{
   # assign params
   my ( $filename, $section ) = @_;

   my $line;      # current line from the file
   my $comments = "";
   my $savesection = 1;

   # make sure there are two arguments (sanity check on the calls)
   ( @_ == 2 ) || die "ReadINISection requires two parameters";

   # attempt to open the file
   open( INIFILE, $filename ) || return 0;

   # search for the section
   $line = <INIFILE>;

   while ( $line )
      {
      # if the sections name is not the same, read it in.
      if ( !( $line =~ /\[$section\]/ ))
         {

         # Dump the line into the string
         $comments = "$comments$line";
         $savesection = 1;
         }
      else
         {
         $savesection = 0;
         }

      # now loop until end of section
      while ( $line = <INIFILE> )
         {
         if ( $line =~ /^\s*\[.*\]/ )  # note:  \s matches white space
            {
            # found a new section, bail out
            last;
            }
         else
            {
            if ( $savesection )
               {
               $comments = "$comments$line";
               }
            } # Section has brackets
         }  # while in the section

      }  # while not eof



   close( INIFILE );

# debug - print what we found
#    my $sKey;
#    foreach $sKey ( keys %pairs )
#       {
#       printf( "'%s'='%s'\n", $sKey, $pairs{ $sKey } );
#       }
#   <STDIN>;

   # return the hash of keys and values
   return $comments;

}  # SkipINISection



# This function takes a ~/path or ~username/path and resolves it to a full path
sub tildeexpand {
        local($_) = @_;
        local($[) = 0;
        s#^~(\w*)#
                (length($1) ?
                        (getpwnam($1))[7] :
                        ($ENV{'HOME'} || $ENV{'LOGDIR'})) ||
                "~$1"
        #e; die $@ if $@;
        "$_";
}



